/*
 * routines to map a myrinet fabric
 */
#include <stdio.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <time.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_xbar32.h"
#include "lf_myri_packet.h"
#include "lf_scheduler.h"
#include "lf_topo_map.h"
#include "lf_fms_comm.h"
#include "lf_lag.h"
#include "libmyri.h"

#include "fma.h"
#include "fma_myri.h"
#include "fma_fms.h"
#include "fma_map.h"
#include "fma_fabric.h"
#include "fma_probe.h"
#include "fma_map_fabric.h"
#include "fma_standalone.h"
#include "fma_sync_timer.h"
#include "fma_settings.h"

/*
 * local prototypes
 */
static void fma_begin_mapping_fabric(
    void (*map_complete_rtn)(struct lf_topo_map *topo_map, int topo_map_size));
static void fma_start_mapping_next_nic_port(void);
static void fma_mf_explore_frontier_for_nics(void);
static void fma_mf_explore_nic_port(struct lf_nic *nicp, int port);
static void fma_mf_scout_node_for_nics(struct fma_nic_info *nip, int p,
                                  union lf_node *np, int port);
static void fma_mf_scout_node_for_xbars(struct fma_nic_info *nip, int p,
                                  union lf_node *np, int port);
static void fma_mf_exploration_done(void);
static void fma_mf_init_xbar_private(struct lf_xbar *xp,
    struct fma_nic_info *nip, int nic_port, unsigned char *route,
    int route_len, int in_port);
static void fma_mf_init_nic_private(struct lf_nic *nicp,
  struct fma_nic_info *nip, int nic_port, unsigned char *route, int route_len);
static void fma_mf_add_nic_to_map(struct lf_nic *nicp);
static struct fma_mf_info *fma_mf_get_mf_info(union lf_node *np, int port);
static void fma_mf_node_port_closed(union lf_node *np, int port);
static void fma_mf_connect_nodes(union lf_node *np1, int port1,
                                 union lf_node *np2, int port2, int resolved);
static void fma_mf_explore_xbar_for_xbars(struct lf_xbar *xp);
static void fma_mf_explore_xbar_for_nics(struct lf_xbar *xp);
static int fma_mf_start_xbar_compares(struct lf_xbar *xp, int port);
static int fma_mf_send_xbar_compares(struct lf_xbar *xp, int port);
static void fma_mf_start_next_xbar_compares(void);
static void fma_mf_compare_xbar_port(struct lf_xbar *xp, int port,
                                     struct lf_xbar *txp);
static void fma_mf_root_nic_scout_timeout(void *v);
static void fma_mf_root_xbar_scout_timeout(void *v);
static void fma_mf_nic_scout_timeout(void *v);
static void fma_mf_xbar_scout_timeout(void *v);
static void fma_mf_compare_timeout(void *vxp);
static void fma_mf_xbar_is_unique(struct lf_xbar *xp);
static void fma_mf_destroy_xbar(struct lf_xbar *xp);
static struct lf_xbar *fma_mf_find_xbar_by_xbar_id(int xbar_id);
static struct lf_xbar *fma_mf_find_xbar_by_node_id(int node_id);
static int fma_mf_merge_xbars(struct lf_xbar *cxp, int cport,
                                       struct lf_xbar *nxp, int nport);
static void fma_mf_unlink_xbar_from_mf_list(struct lf_xbar *xp);
static void fma_mf_stop_compares(struct lf_xbar *xp);
static int fma_mf_send_stage0_compares(struct lf_xbar *xp, int port);
static int fma_mf_send_stage1_compares(struct lf_xbar *xp, int port);
static int fma_mf_send_stage2_compares(struct lf_xbar *xp, int port);
static void fma_mf_mapping_complete(void);
static void fma_mf_make_topo_map(void);
static struct lf_fabric *fma_mf_create_fabric(void);
static void fma_mf_restart_mapping(void);
static void fma_mf_free_mapping(void);
static struct lf_nic *fma_mf_find_nic_by_mac(struct fma_map_progress *mpp,
                                             lf_mac_addr_t mac_addr);
static struct lf_host *fma_mf_get_host_by_max_mac(lf_mac_addr_t mac);
static void fms_mf_normalize_xbar(struct lf_xbar *xp);
static void fma_mf_calculate_forwarding(int *idp, int *port);
static void fma_mf_done_comparing(void);
static void fma_mf_fill_xbar_map_scout(struct fma_xbar_scout_pkt *pkt,
  int node_index, int is_tagged);
static void fma_mf_fill_nic_map_scout(struct myri_nic_scout *pkt,
  int node_index, unsigned char *route, int route_len);
static void fma_mf_remove_nascent_placeholders(void);
static void fma_mf_create_nascent_xbar(union lf_node *fnp, int fport);
static void fma_mf_add_xbar_to_list(struct fma_xbar *anchor,
    struct lf_xbar *xp);
#define fma_mf_add_xbar_to_resolving(XP) \
  fma_mf_add_xbar_to_list(&A.myri->map_progress->resolving_xbar_anchor, XP)
#define fma_mf_add_xbar_to_frontier(XP) \
  fma_mf_add_xbar_to_list(&A.myri->map_progress->frontier_xbar_anchor, XP)
#define fma_mf_add_xbar_to_resolved(XP) \
  fma_mf_add_xbar_to_list(&A.myri->map_progress->resolved_xbar_anchor, XP)
#define fma_mf_add_xbar_to_nascent(XP) \
  fma_mf_add_xbar_to_list(&A.myri->map_progress->nascent_xbar_anchor, XP)


#define mf_debug (A.debug>1)
#define mf_debug2 (A.debug>2)

static void dump_map(void);

#ifdef FM_TEST_LAG
extern char *My_lag_id;
#endif


/*
 * Given a MAC address, see if any host has reported it as their
 * max mac address and return the index if so.  If not, create a new
 * host entry and return that.
 * This is only called on new NICs, so we know we need to increment the 
 * num_nics field of the host record.
 */
static struct lf_host *
fma_mf_get_host_by_max_mac(
  lf_mac_addr_t host_max_mac)
{
  struct fma_map_progress *mpp;
  struct lf_host *hp;
  struct fma_host *fhp;
  int h;

  mpp = A.myri->map_progress;

  /* find it if we can */
  for (h=0; h<mpp->num_hosts; ++h) {
    hp = mpp->mf_hosts[h];
    if (LF_MAC_CMP(host_max_mac, FMA_HOST(hp)->max_mac_addr) == 0) {
      return hp;
    }
  }

  /*
   * We have a new MAC address.  First, make sure there is room
   * in the table and grow it if not.
   */
  if (h <= mpp->host_table_size) {
    mpp->host_table_size += 100;
    mpp->mf_hosts = (struct lf_host **) realloc(mpp->mf_hosts,
	sizeof(struct lf_host *) * mpp->host_table_size);
    if (mpp->mf_hosts == NULL) {
      LF_ERROR(("Error growing mpp->mf_hosts"));
    }
  }

  /* copy in the mac address */
  hp = fma_create_host();
  mpp->mf_hosts[h] = hp;
  fhp = FMA_HOST(hp);;

  /* copy in max mac addr */
  LF_MAC_COPY(fhp->max_mac_addr, host_max_mac);
  ++mpp->num_hosts;
  
  return hp;

 except:
  fma_perror_exit(1);
  return NULL;
}

/*
 * Something went awry with our attempt at mapping, start over
 */
static void
fma_mf_restart_mapping()
{
  void (*save_rtn)(struct lf_topo_map *topo_map, int topo_map_size);

  fma_log("Restarting mapping.");

  /* save completion  routine */
  save_rtn = A.myri->map_progress->map_complete_rtn;

  /* free all the data for the mapping process */
  fma_mf_cancel_mapping();

  /* Restart mapping process */
  fma_begin_mapping_fabric(save_rtn);
}

static void
fma_mf_increment_map_session_id()
{
  struct fma_myri *fmp;

  fmp = A.myri;

  /* Increment session ID (but not zero) */
  do {
    ++fmp->map_session_id;
  } while (fmp->map_session_id == 0);
}

/*
 * Generate a map of the fabric using node and xbar queries
 */
void
fma_start_mapping_fabric(
  void (*map_complete_rtn)(struct lf_topo_map *topo_map, int topo_map_size))
{
  struct fma_map_info *mip;

  /* reset the restart reason */
  mip = A.map_info;
  mip->mi_rest_reason = MIR_NO_REASON;

  /* start the mapping process */
  fma_begin_mapping_fabric(map_complete_rtn);
}

/*
 * This is the common, local entry point for starting the mapping process.
 * Separated out so it can be called by the restart routine.
 */
static void
fma_begin_mapping_fabric(
  void (*map_complete_rtn)(struct lf_topo_map *topo_map, int topo_map_size))
{
  struct fma_myri *fmp;
  struct fma_nic_info *nip;
  struct fma_map_progress *mpp;
  struct lf_host *hp;
  int rc;
  int n;
  int p;

  fmp = A.myri;

  /* don't start again if one mapping already in progress */
  if (fmp->map_progress != NULL || A.stand->mapping_in_progress) {
    if (A.debug) fma_log("Ignoring map start request");
    return;
  }

  /* Release any current topo map */
  fma_release_topo_map();

  fma_log("Mapping fabric...");
  A.stand->mapping_in_progress = TRUE;

  /* start the new map, allocate and fill in a progress struct */
  LF_CALLOC(mpp, struct fma_map_progress, 1);
  fmp->map_progress = mpp;	/* save away map progress pointer */
  LF_CALLOC(mpp->my_nics, struct lf_nic *, fmp->num_nics);

  /* record start time */
  mpp->mf_start_time = time(NULL);

  /* clear the list anchors */
  mpp->resolved_xbar_anchor.mf_next = &mpp->resolved_xbar_anchor;
  mpp->resolved_xbar_anchor.mf_prev = &mpp->resolved_xbar_anchor;
  mpp->resolving_xbar_anchor.mf_next = &mpp->resolving_xbar_anchor;
  mpp->resolving_xbar_anchor.mf_prev = &mpp->resolving_xbar_anchor;
  mpp->frontier_xbar_anchor.mf_next = &mpp->frontier_xbar_anchor;
  mpp->frontier_xbar_anchor.mf_prev = &mpp->frontier_xbar_anchor;
  mpp->nascent_xbar_anchor.mf_next = &mpp->nascent_xbar_anchor;
  mpp->nascent_xbar_anchor.mf_prev = &mpp->nascent_xbar_anchor;
  mpp->nic_anchor.mf_next = &mpp->nic_anchor;
  mpp->nic_anchor.mf_prev = &mpp->nic_anchor;

  /* save completion routine */
  mpp->map_complete_rtn = map_complete_rtn;

  /* clear mapped flags on all our NICs */
  for (n=0; n<fmp->num_nics; ++n) {
    struct myri_nic_info *mip;
    struct lf_nic *nicp;

    nip = fmp->nic_info[n];
    mip = &nip->myri_info;

    for (p=0; p<mip->num_ports; ++p) {
      nip->mf_mapped[p] = FALSE;
    }

    /* create the NIC struct */
    nicp = fma_create_nic(mip->num_ports, mip->mac_addr, myri_firmware_type());
    if (nicp == NULL) LF_ERROR(("Error creating NIC"));
    mpp->my_nics[n] = nicp;
    FMA_NIC(nicp)->local_nip = nip;
    nicp->host_nic_id = nip->nic_id;

    /* Fill FMA-private structs */
    fma_mf_init_nic_private(mpp->my_nics[n], NULL, 0, NULL, 0);

    fma_mf_add_nic_to_map(mpp->my_nics[n]);	/* add it to the map */

    /* save these away in the NIC even though they are really host values */
    FMA_NIC(nicp)->fma_flags = A.my_fma_flags;
    FMA_NIC(nicp)->fw_type = myri_firmware_type();

#ifdef FM_TEST_LAG
    memset(nicp->nic_lag_id, 0, 26);
    if (My_lag_id != NULL) {
      strcpy(nicp->nic_lag_id, My_lag_id);
    }
#endif

    /* Create and assign host entry */
    hp = fma_mf_get_host_by_max_mac(fmp->my_max_mac);
    rc = lf_add_existing_nic_to_host(nicp, hp);
    if (rc == -1) LF_ERROR(("Error adding NIC to host"));

    /* fill in with our hostname */
    LF_DUP_STRING(hp->hostname, A.hostname);
  }

  /* now, start mapping on the next next NIC */
  fma_start_mapping_next_nic_port();

  return;

 except:
  if (mpp != NULL) LF_FREE(mpp->my_nics);
  LF_FREE(fmp->map_progress);
  fma_perror_exit(1);
}

static void
fma_start_mapping_next_nic_port()
{
  int n;
  struct fma_map_progress *mpp;
  struct fma_myri *fmp;

  fmp = A.myri;
  mpp = fmp->map_progress;

  for (n=0; n<fmp->num_nics; ++n) {
    struct fma_nic_info *nip;
    int p;

    nip = fmp->nic_info[n];
    for (p=0; p<nip->myri_info.num_ports; ++p) {
      if (!nip->mf_mapped[p]) {
	nip->mf_mapped[p] = TRUE;
	fma_mf_explore_nic_port(mpp->my_nics[n], p);
	return;
      }
    }
  }

  if (mf_debug) fma_log("no more NIC ports to map!");

  fma_mf_mapping_complete();		/* mapping is all done */
}

/*
 * Exploration is all done.  Now, we need to start the resolving process.
 * At this point, there should be no "unknown" ports on any frontier xbar.
 * Normalize each frontier xbar and let resolving begin.
 */
static void
fma_mf_exploration_done()
{
  struct fma_map_progress *mpp;
  struct fma_xbar *frontier;
  struct fma_xbar *fxp;
  struct lf_xbar *xp;
  int port;

  mpp = A.myri->map_progress;

  if (mf_debug) fma_log("Exploration phase complete");

  /* normalize all untagged frontier xbars and
   * Mark all disconnected ports on frontier xbars as "closed"
   */
  frontier = &mpp->frontier_xbar_anchor;
  for (fxp=frontier->mf_next; fxp!=frontier; fxp=fxp->mf_next) {
    xp = fxp->parent;

    /* Mark all disconnected ports closed */
    for (port=0; port<xp->num_ports; ++port) {
      if (xp->topo_ports[port] == NULL) {
	fma_mf_node_port_closed(LF_NODE(xp), port);

	if (mf_debug) {
	  fma_log("x%d:p%d is disconnected", FMA_XBAR(xp)->node_id, port);
	}
      }
    }

    /* normalize anonymous xbars */
    if (xp->xbar_id == 0){
      fms_mf_normalize_xbar(fxp->parent);
    }
  }

  /* start xbar compares */
  fma_mf_start_next_xbar_compares();
}

/*
 * Shift the ports on an xbar down so that 0 is always connected.
 * Also, compute port-connect signature.
 */
static void
fms_mf_normalize_xbar(
  struct lf_xbar *xp)
{
  struct fma_mf_info *mfip;
  int signature;
  int start;
  int p;

  /* find first connected port */
  for (p=0; p<xp->num_ports; ++p) {
    if (xp->topo_ports[p] != NULL) break;
  }

  /* If no connected ports, all done */
  if (p >= xp->num_ports) {
    FMA_XBAR(xp)->mf_signature = 0;
    return;
  }

  /*
   * If needed, move ports down so that 0 is first.  Move ports, copy 
   * mf_info, adjust mf_in_port
   */
  if (p != 0) {

    start = p;
    mfip = FMA_XBAR(xp)->mf_info;

    if (mf_debug && start != 0) fma_log("x%d normalize by %d",
	FMA_XBAR(xp)->node_id, -start);

    FMA_XBAR(xp)->mf_normal_delta = -start;
    FMA_XBAR(xp)->mf_in_port -= start;

    for (p=start; p<xp->num_ports; ++p) {
      union lf_node *onp;
      int newport;

      newport = p-start;

      /* copy over mf_info */
      mfip[newport].mf_closed = mfip[p].mf_closed;
      mfip[p].mf_closed = TRUE;

      onp = xp->topo_ports[p];

      /* If connected, modify both ends of the link */
      if (onp != NULL) {
	int oport;

	oport = xp->topo_rports[p];

	xp->topo_ports[newport] = onp;
	xp->topo_rports[newport] = oport;
	xp->topo_ports[p] = NULL;

	if (onp != FMA_MF_NASCENT_XBAR) {
	  if (onp->ln_type == LF_NODE_NIC) {
	    LF_NIC(onp)->topo_rports[oport] = newport;
	  } else {
	    LF_XBAR(onp)->topo_rports[oport] = newport;
	  }
	}
      }
    }
  }

  /* compute signature */
  signature = 0;
  for (p=0; p<xp->num_ports; ++p) {
    if (xp->topo_ports[p] != NULL) {
      signature |= (1 << p);
    }
  }

  /* check for overlarge sig with -x specified */
  if (A.xbar_types == FMA_XT_NO_IDS && signature > 0xFFFF) {
    LF_ERROR(("Signature for x%d is %x,"
	      "likely -x used in fabric with tagged xbars", 
	      FMA_XBAR(xp)->node_id, signature));
  }

  /* assign signture */
  FMA_XBAR(xp)->mf_signature = signature;

  if (mf_debug) fma_log("Signature for x%d is %08x", FMA_XBAR(xp)->node_id,
      signature);
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Explore one of our local ports.  First, we will scout for another NIC.
 */
static void
fma_mf_explore_nic_port(
  struct lf_nic *nicp,
  int port)
{
  struct fma_map_progress *mpp;
  struct fma_settings *asp;

  mpp = A.myri->map_progress;
  asp = A.settings;

  fma_mf_increment_map_session_id();	/* bump session ID */

  /* Save NIC and port we are now mapping */
  mpp->mf_nic_being_mapped = nicp;
  mpp->mf_port_being_mapped = port;

  mpp->mf_scout_retries_left = asp->nic_scout_retries + 1;
  fma_mf_root_nic_scout_timeout(NULL);
}

/*
 * root NIC scout has timed out.  The fact that we are here means no
 * response has been received and the port is still NULL, since we cancel
 * the timer when we fill in the port.
 */
static void
fma_mf_root_nic_scout_timeout(
  void *v)
{
  struct fma_map_progress *mpp;
  struct lf_nic *nicp;
  struct fma_settings *asp;
  int port;

  mpp = A.myri->map_progress;
  asp = A.settings;
  nicp = mpp->mf_nic_being_mapped;
  port = mpp->mf_port_being_mapped;

  mpp->mf_scout_timer = NULL;		/* popped, so clear it */
  mpp->mf_outstanding_scouts = 0;	/* scout no longer outstanding */

  /* If any retries left, re-send the scout */
  if (mpp->mf_scout_retries_left > 0) {
    --mpp->mf_scout_retries_left;

    /* resend a NIC scout out this port */
    fma_mf_scout_node_for_nics(FMA_NIC(nicp)->local_nip, port,
	LF_NODE(nicp), port);

    /* start a timer if anything sent */
    if (mpp->mf_outstanding_scouts > 0) {

      /* start a sync timer for the nic scout */
      mpp->mf_scout_timer = fma_sync_timer_setup(
		FMA_NIC(nicp)->local_nip->nic_handle, port,
		fma_mf_root_nic_scout_timeout, NULL, asp->nic_scout_timeout);
      if (mpp->mf_scout_timer == NULL) {
	LF_ERROR(("Error setting up root nic scout timer"));
      }
    } else {
      fma_mf_exploration_done();
    }

  /* If no retries left, proceed to scouting for an xbar */
  } else {

    mpp->mf_scout_retries_left = FMA_XBAR_SCOUT_RETRY+1;
    fma_mf_root_xbar_scout_timeout(NULL);
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * root xbar scout(s) have timed out.  there might be an anonymous xbar
 * attached, in which case we only need to restart the scouts for tagged
 * xbars.  If we got a tagged xbar response already, then the timer will
 * be cleared and we never get here.
 */
static void
fma_mf_root_xbar_scout_timeout(
  void *v)
{
  struct fma_map_progress *mpp;
  struct lf_nic *nicp;
  int port;

  mpp = A.myri->map_progress;
  nicp = mpp->mf_nic_being_mapped;
  port = mpp->mf_port_being_mapped;

  mpp->mf_scout_timer = NULL;	/* popped, so clear it */
  mpp->mf_outstanding_scouts = 0;	/* scouts no longer outstanding */

  /* If any retries left, re-send the scout */
  if (mpp->mf_scout_retries_left > 0) {
    --mpp->mf_scout_retries_left;

    /* send xbar scouts out this port */
    fma_mf_scout_node_for_xbars(FMA_NIC(nicp)->local_nip, port,
	LF_NODE(nicp), port);

    /* start a timer if anything sent */
    if (mpp->mf_outstanding_scouts > 0) {

      /* start a sync timer for the xbar scout */
      mpp->mf_scout_timer = fma_sync_timer_setup(
		FMA_NIC(nicp)->local_nip->nic_handle, port,
		fma_mf_root_xbar_scout_timeout, NULL, FMA_XBAR_SCOUT_TIMEOUT);
      if (mpp->mf_scout_timer == NULL) {
	LF_ERROR(("Error setting up root xbar scout timer"));
      }
    }
  }

  /* If no more retries, or nothing got sent, move on to next phase */
  if (mpp->mf_outstanding_scouts <= 0) {
    fma_mf_exploration_done();
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Scout a frontier port for a NIC.
 */
static void
fma_mf_scout_node_for_nics(
  struct fma_nic_info *nip,
  int p,
  union lf_node *np,
  int port)
{
  unsigned char route[FMA_IFC_ROUTE_LEN*2+1];
  int route_len;
  struct fma_map_progress *mpp;
  struct myri_nic_scout scout;
  int node_index;
  int rc;

  mpp = A.myri->map_progress;

  /*
   * If coming from a NIC, route is empty, else it's the route to this
   * xbar plus a byte to leave by the appropriate port
   */
  if (np->ln_type == LF_NODE_NIC) {

    /* if something already connected, skip it */
    if (LF_NIC(np)->topo_ports[port] != NULL) {
      return;
    }

    route_len = 0;
    node_index = LF_TOPO_NODE_INDEX(nip->nic_id, LF_TOPO_NIC, p);

  } else if (np->ln_type == LF_NODE_XBAR) {
    struct fma_xbar *fxp;

    /* if something already connected, skip it */
    if (LF_XBAR(np)->topo_ports[port] != NULL) {
      return;
    }

    /* add one byte to route to this xbar */
    fxp = FMA_XBAR_N(np);
    route_len = fxp->mf_route_len+1;
    memcpy(route, fxp->mf_route, fxp->mf_route_len);
    route[route_len-1] = LF_DELTA_TO_ROUTE(port - fxp->mf_in_port);

    node_index = LF_TOPO_NODE_INDEX(fxp->node_id, LF_TOPO_XBAR, port);
    
  } else {
    LF_ERROR(("Bad node type"));
  }

  /* Build the NIC scout packet */
  fma_mf_fill_nic_map_scout(&scout, node_index, route, route_len);

  ++mpp->mf_outstanding_scouts;

  /* send the packet */
  rc = fma_myri_raw_send(nip->nic_id, p, route, route_len,
      &scout, sizeof(scout), NULL, NULL);
  if (rc == -1) LF_ERROR(("Error sending NIC scout"));

  return;

 except:
  dump_map();
  fma_perror_exit(1);
}

/*
 * Send xbar scouts for a frontier port - scout for tagged xbar and anonymous
 * xbar in that order.  If an xbar is tagged, we want the ID back before
 * the simple bounce.
 */
static void
fma_mf_scout_node_for_xbars(
  struct fma_nic_info *nip,
  int p,
  union lf_node *np,
  int port)
{
  unsigned char route[FMA_IFC_ROUTE_LEN*2+1];
  struct fma_map_progress *mpp;
  struct fma_xbar_scout_pkt scout;
  struct lf_nic *nicp;
  union lf_node *cnp;
  int loop_route_len;
  int do_anonymous;
  int node_index;
  int route_len;
  int rc;

  /*
   * First of all, check port for existing connections.  This controls whether
   * we scout for both types of xbars, just tagged, or none at all
   */
  if (np->ln_type == LF_NODE_NIC) {
    cnp = LF_NIC(np)->topo_ports[port];
  } else {
    cnp = LF_XBAR(np)->topo_ports[port];
  }
  if (cnp == NULL) {
    do_anonymous = TRUE;
  } else if (cnp == FMA_MF_NASCENT_XBAR) {
    do_anonymous = FALSE;
  } else if (cnp->ln_type == LF_NODE_XBAR && LF_XBAR(cnp)->xbar_id == 0) {
    do_anonymous = FALSE;

  /* if we get here, either a NIC or tagged xbar, so no scout needed */
  } else {
    return;
  }

  mpp = A.myri->map_progress;

  /*
   * Compute route out the port we are scouting.
   * If coming from a NIC, route is empty, else it's the route to this
   * xbar plus a byte to leave by the appropriate port
   */
  if (np->ln_type == LF_NODE_NIC) {
    nicp = LF_NIC(np);

    route_len = 0;
    node_index = LF_TOPO_NIC_INDEX(FMA_NIC(nicp)->local_nip->nic_id, p);

  } else if (np->ln_type == LF_NODE_XBAR) {
    struct fma_xbar *fxp;

    /* add one byte to route to this xbar */
    fxp = FMA_XBAR_N(np);
    route_len = fxp->mf_route_len+1;
    memcpy(route, fxp->mf_route, fxp->mf_route_len);
    route[route_len-1] = LF_DELTA_TO_ROUTE(port - fxp->mf_in_port);

    node_index = LF_TOPO_XBAR_INDEX(fxp->node_id, port);
    
  } else {
    LF_ERROR(("Bad node type"));
  }

  /* convert route to a loop route */
  loop_route_len = lf_loop_route(route, route, route_len,
      LF_TAGGED_XBAR_QUERY_ROUTE);

  /*
   * start the tagged xbar scout unless we know there are none
   */
  if (A.xbar_types != FMA_XT_NO_IDS) {

    ++mpp->mf_outstanding_scouts;

    /* Build a tagged xbar scout packet */
    fma_mf_fill_xbar_map_scout(&scout, node_index, TRUE);

    /* send the packet */
    rc = fma_myri_raw_send(nip->nic_id, p, route, loop_route_len,
	&scout, sizeof(scout), NULL, NULL);
    if (rc == -1) LF_ERROR(("Error sending tagged xbar scout"));
  }

  /*
   * start the anonymous xbar scout unless we know there are none
   */
  if (A.xbar_types != FMA_XT_ID_ONLY && do_anonymous) {

    ++mpp->mf_outstanding_scouts;

    /* fill in a generic loop byte */
    route[route_len] = LF_DELTA_TO_ROUTE(0);

    /* Build an anonymous xbar scout packet */
    fma_mf_fill_xbar_map_scout(&scout, node_index, FALSE);

    /* send the packet */
    rc = fma_myri_raw_send(nip->nic_id, p, route, loop_route_len,
	&scout, sizeof(scout), NULL, NULL);
    if (rc == -1) LF_ERROR(("Error sending anonymous xbar scout"));
  }

  return;

 except:
  dump_map();
  fma_perror_exit(1);
}

/*
 * Fill in the MF-private fields of an xbar
 */
static void
fma_mf_init_xbar_private(
  struct lf_xbar *xp,
  struct fma_nic_info *nip,
  int nic_port,
  unsigned char *route,
  int route_len,
  int in_port)
{
  struct fma_mf_info *mfip;

  /* array of flags for recording xbar scout responses */
  LF_CALLOC(mfip, struct fma_mf_info, xp->num_ports);
  FMA_XBAR(xp)->mf_info = mfip;

  /* record which NIC to start from for this xbar */
  FMA_XBAR(xp)->mf_nic = nip;
  FMA_XBAR(xp)->mf_nic_port = nic_port;

  /* copy over route info */
  memcpy(FMA_XBAR(xp)->mf_route, route, route_len);
  FMA_XBAR(xp)->mf_route_len = route_len;
  FMA_XBAR(xp)->mf_in_port = in_port;
  return;

 except:
  dump_map();
  fma_perror_exit(1);
}

/*
 * Fill in the MF-private fields of a NIC
 */
static void
fma_mf_init_nic_private(
  struct lf_nic *nicp,
  struct fma_nic_info *nip,
  int nic_port,
  unsigned char *route,
  int route_len)
{
  struct fma_mf_info *mfip;

  /* array of flags for recording xbar scout responses */
  LF_CALLOC(mfip, struct fma_mf_info, nicp->num_ports);
  FMA_NIC(nicp)->mf_info = mfip;

  /* record which of our NICs to start from for this NIC */
  FMA_NIC(nicp)->mf_nic = nip;
  FMA_NIC(nicp)->mf_nic_port = nic_port;

  /* copy over route info */
  memcpy(FMA_NIC(nicp)->mf_route, route, route_len);
  FMA_NIC(nicp)->mf_route_len = route_len;
  return;

 except:
  dump_map();
  fma_perror_exit(1);
}

/*
 * Add a NIC to the map we are creating
 */
static void
fma_mf_add_nic_to_map(
  struct lf_nic *nicp)
{
  struct fma_nic *fnp;
  struct fma_nic *anchor;

  anchor = &A.myri->map_progress->nic_anchor;
  fnp = FMA_NIC(nicp);

  /* Add this NIC to the map list */
  fnp->mf_next = anchor->mf_next;
  fnp->mf_prev = anchor;
  fnp->mf_next->mf_prev = fnp;
  fnp->mf_prev->mf_next = fnp;

  return;
}

/*
 * Add an xbar to the map we are creating
 */
static void
fma_mf_add_xbar_to_list(
  struct fma_xbar *anchor,
  struct lf_xbar *xp)
{
  struct fma_xbar *fxp;

  fxp = FMA_XBAR(xp);

  /* Add this xbar to the map list */
  fxp->mf_next = anchor->mf_next;
  fxp->mf_prev = anchor;
  fxp->mf_next->mf_prev = fxp;
  fxp->mf_prev->mf_next = fxp;

  return;
}

/*
 * Get fabric mapping info for this node
 */
static struct fma_mf_info *
fma_mf_get_mf_info(
  union lf_node *np,
  int port)
{
  if (np->ln_type == LF_NODE_XBAR) {
    return FMA_XBAR_N(np)->mf_info + port;
  } if (np->ln_type == LF_NODE_NIC) {
    return FMA_NIC_N(np)->mf_info + port;
  } else {
    LF_ERROR(("Bad node type"));
  }
 except:
  dump_map();
  fma_perror_exit(1);
  return NULL;
}


/*
 * Mark a port on this node as closed
 */
static void
fma_mf_node_port_closed(
  union lf_node *np,
  int port)
{
  struct fma_mf_info *mfip;

  mfip = fma_mf_get_mf_info(np, port);

  /* mark this port as resolved */
  mfip->mf_closed = TRUE;
}

/*
 * Look for a NIC in the map
 */
static struct lf_nic *
fma_mf_find_nic_by_mac(
  struct fma_map_progress *mpp,
  lf_mac_addr_t mac_addr)
{
  struct fma_nic *fnicp;

  /* loop through all NICs */
  for (fnicp=mpp->nic_anchor.mf_next;
       fnicp != &mpp->nic_anchor;
       fnicp=fnicp->mf_next) {
    if (LF_MAC_CMP(fnicp->parent->mac_addr, mac_addr) == 0) {
      return fnicp->parent;
    }
  }

  return NULL;
}

/*
 * Got a response from a NIC scout.
 * If this is not one of our NICs, add it to the map.
 * In any case, make a connection from this frontier node to the NIC.
 * If this NIC was already connected, merge this xbar into the xbar the NIC
 * is already connected to.
 */
void
fma_mf_got_nic_scout_resp(
  struct myri_nic_scout_reply *pkt,
  int len)
{
  struct fma_nic_map_scout_opaque_data *sodp;
  struct fma_nic_reply_opaque_data *rodp;
  struct fma_mf_info *mfip;
  struct fma_nic_info *nip;
  struct fma_map_info *mip;
  struct fma_myri *fmp;
  struct fma_map_progress *mpp;
  struct lf_nic *nnicp;
  struct lf_host *hp;
  struct lf_xbar *fxp;
  union lf_node *fnp;
  unsigned char route[FMA_IFC_ROUTE_LEN];
  int node_index;
  int nic_port;
  int node_type;
  int route_len;
  int node_id;
  int fport;
  int nport;
  int rc;
  int n;

  fmp = A.myri;
  mpp = fmp->map_progress;

  /* just return if mapping cancelled */
  if (mpp == NULL) return;

  sodp = (struct fma_nic_map_scout_opaque_data *) &pkt->opaque_scout_data;
  rodp = (struct fma_nic_reply_opaque_data *) &pkt->opaque_reply_data;
  nip = FMA_NIC(mpp->mf_nic_being_mapped)->local_nip;
  nic_port = mpp->mf_port_being_mapped;

  /* Ignore packets from previous scout sessions */
  if (ntohl(sodp->serial_32) != fmp->map_session_id) {
    return;
  }

  /* Ignore packets that make us croak */
  if (pkt->num_ports_8 < 1) {
    if (A.debug) {
      fma_log("Ignoring corrupt packet from %s",
	      fma_mac_to_hostname(pkt->mac_addr));
    }
    return;
  }

  /* find the node we were scouting from */
  node_index = ntohl(sodp->node_index_32);
  node_type = LF_TOPO_NI_TYPE(node_index);
  fport = LF_TOPO_NI_PORT(node_index);
  node_id = LF_TOPO_NI_INDEX(node_index);
  nport = pkt->port_8;

  if (node_type == LF_TOPO_XBAR) {

    /* follow forwarding if present - NIc replies are precious */
    fma_mf_calculate_forwarding(&node_id, &fport);

    fxp = fma_mf_find_xbar_by_node_id(node_id);
    if (fxp == NULL) {
      if (mf_debug) {
	fma_log("Dropping NIC scout of stale xbar x%d", node_id); 
      }
      return;
    }

    fnp = LF_NODE(fxp);

    /* add one byte to route to this xbar */
    route_len = FMA_XBAR(fxp)->mf_route_len+1;
    memcpy(route, FMA_XBAR(fxp)->mf_route, FMA_XBAR(fxp)->mf_route_len);
    route[route_len-1] = LF_DELTA_TO_ROUTE(fport - FMA_XBAR(fxp)->mf_in_port);

  } else {
    fnp = LF_NODE(mpp->mf_nic_being_mapped);
    route_len = 0;
    fxp = NULL;
  }

  if (mf_debug) {
    lf_string_t s;
    int off;

    off = sprintf(s, "got resp from NIC on %s (fw type 0x%x),",
	rodp->nr_hostname, pkt->firmware_type_8);
    off += sprintf(s+off, LF_MAC_FORMAT " port %d on ",
	LF_MAC_ARGS(pkt->mac_addr), pkt->port_8);
    if (fnp->ln_type == LF_NODE_NIC) {
      off += sprintf(s+off, "NIC");
    } else {
      off += sprintf(s+off, "x%d", FMA_XBAR_N(fnp)->node_id);
    }
    off += sprintf(s+off, ":%d", fport);
    fma_log(s);
  }

  /* see if this is one of our NIC ports */
  nnicp = NULL;
  for (n=0; n<fmp->num_nics; ++n) {
    if (LF_MAC_CMP(mpp->my_nics[n]->mac_addr,
	           pkt->mac_addr) == 0) {
      struct fma_nic_info *nip;

      nip = fmp->nic_info[n];
      nnicp = mpp->my_nics[n];

      nip->mf_mapped[nport] = TRUE;
      break;
    }
  }

  /* if not one of ours, see if it is already known to us */
  if (nnicp == NULL) {

    /*
     * tell standalone that we had contact.  This may cause mapping to
     * be cancelled, so recheck that map_progress is not NULL.  If it is,
     * just return now.
     */
    if (A.run_state == FMA_RUN_STANDALONE) {
      fma_standalone_got_contact(rodp->level_8, rodp->max_mac_addr,
				 nip, nic_port, route, route_len,
				 FMA_NST_VERIFY);

      /* if map_progress got NULLED, all done */
      if (fmp->map_progress == NULL) return;

    }

    /* Look for this NIC in our map */
    nnicp = fma_mf_find_nic_by_mac(mpp, pkt->mac_addr);

    /* If not already in the map, add this NIC */
    if (nnicp == NULL) {
      /* create the NIC struct */
      nnicp = fma_create_nic(pkt->num_ports_8,
		      pkt->mac_addr, pkt->firmware_type_8);
      if (nnicp == NULL) LF_ERROR(("Creating NIC"));

      /* Fill FMA-private structs */
      fma_mf_init_nic_private(nnicp, nip, nic_port, route, route_len);

      fma_mf_add_nic_to_map(nnicp);	/* add it to the map */

      /* pull data from the query reply */
      nnicp->host_nic_id = ntohs(rodp->host_nic_id_16);
      FMA_NIC(nnicp)->fma_flags = ntohs(rodp->fma_flags_16);
      lf_lag_copy(nnicp->nic_lag_id, rodp->nr_lag_id);

      /* get parent host */
      hp = fma_mf_get_host_by_max_mac(rodp->max_mac_addr);
      rc = lf_add_existing_nic_to_host(nnicp, hp);
      if (rc == -1) LF_ERROR(("Error adding NIC to host"));

      /* copy over the hostname */
      LF_DUP_STRING(hp->hostname, (char *)rodp->nr_hostname);
    }
  }


  /*
   * If frontier is a NIC, just connect them.
   */
  if (fnp->ln_type == LF_NODE_NIC) {
    fma_mf_connect_nodes(fnp, fport, LF_NODE(nnicp), nport, TRUE);

  /*
   * Otherwise frontier is xbar, more to do...
   */
  } else {

    /* If not already connected, link frontier xbar to the NIC.
     * If the NIC *is* already connected, then merge that xbar into this
     * one.
     */
    if (nnicp->topo_ports[nport] == NULL) {

      /*
       * If this port is "closed", then this is a really late NIC
       * reply, and will mess us up if we use it.
       */
      mfip = FMA_XBAR(fxp)->mf_info + fport;
      if (mfip->mf_closed) {
	if (mf_debug) {
	  fma_log("Ignoring NIC reply on closed xbar port");
	}
	return;
      }

      /* connect this NIC to frontier node */
      fma_mf_connect_nodes(fnp, fport, LF_NODE(nnicp), nport, TRUE);

      /* Finding a NIC makes us resolved, just mark it for now */
      FMA_XBAR_N(fnp)->mf_xbar_resolved = TRUE;

    } else {
      struct lf_xbar *fxp2;
      int fport2;

      fxp2 = LF_XBAR(nnicp->topo_ports[nport]);
      fport2 = nnicp->topo_rports[nport];

      /* If we are already connected to this xbar, DUP reply, ignore */
      if (fxp2 == fxp) {
	if (fport2 != fport) {
	  mip = A.map_info;

	  /* only report as DUP if we get same error twice on same NIC */
	  if (mip->mi_rest_reason == MIR_DUP_MAC_PORT &&
	      LF_MAC_CMP(pkt->mac_addr, mip->mi_rest_mac_addr) == 0) {
	    errno = 0;
	    fma_log("Same NIC attached to 2 different ports - DUP MAC addr?");
	    LF_ERROR(("MAC address is " LF_MAC_FORMAT,
		LF_MAC_ARGS(pkt->mac_addr)));

	  /* Might be due to packet loss, retry mapping */
	  } else {
	    fma_log("Restart mapping due to NIC port conflict");
	    LF_MAC_COPY(mip->mi_rest_mac_addr, pkt->mac_addr);
	    mip->mi_rest_reason = MIR_DUP_MAC_PORT;
	    goto restart;
	  }
	}
	return;
      }

      if (mf_debug) fma_log("Merging x%d into x%d -- common NIC",
	  FMA_XBAR(fxp2)->node_id, FMA_XBAR(fxp)->node_id);

      /* merge fxp into fxp2 */
      rc = fma_mf_merge_xbars(fxp, fport, fxp2, fport2);
      if (rc == -1) {
	mip = A.map_info;

	/* only report as DUP if we get same error twice on same NIC */
	if (mip->mi_rest_reason == MIR_DUP_MAC_XBAR &&
	    LF_MAC_CMP(pkt->mac_addr, mip->mi_rest_mac_addr) == 0) {
	  errno = 0;
	  fma_log("Same NIC attached to 2 different xbars - DUP MAC addr?");
	  LF_ERROR(("MAC address is " LF_MAC_FORMAT,
	      LF_MAC_ARGS(pkt->mac_addr)));

	/* Might be due to packet loss, retry mapping */
	} else {
	  fma_log("Restart mapping due to NIC xbar conflict");
	  LF_MAC_COPY(mip->mi_rest_mac_addr, pkt->mac_addr);
	  mip->mi_rest_reason = MIR_DUP_MAC_XBAR;
	  goto restart;
	}
      }
    }
  }
  return;

 restart:
  fma_mf_restart_mapping();
  return;

 except:
  if (A.debug) {
    dump_map();
  }
  fma_perror_exit(1);
}

/*
 * Increment the next node_id to be assigned.  Make sure there is
 * always enough room in forwarding address array
 */
static void
fma_mf_increment_node_id()
{
  struct fma_map_progress *mpp;

  mpp = A.myri->map_progress;
  ++mpp->next_node_id;			/* bump the node_id */

  /* If this node_id is not a good index, need to grow table */
  if (mpp->next_node_id >= mpp->xfa_size) {
    int newsize;
    int i;

    /* grow table by 256 entries */
    newsize = mpp->xfa_size + 256;

    if (mf_debug) fma_log("growing forwarding table to %d", newsize);

    mpp->xbar_forward_addresses = (int *)
      realloc(mpp->xbar_forward_addresses, newsize * sizeof(int *));
    if (mpp->xbar_forward_addresses == NULL) {
      LF_ERROR(("Error allocating forwarding array"));
    }
    mpp->xbar_forward_deltas = (int *)
      realloc(mpp->xbar_forward_deltas, newsize * sizeof(int *));
    if (mpp->xbar_forward_deltas == NULL) {
      LF_ERROR(("Error allocating forwarding deltas array"));
    }

    /* An empty slot is "-1" */
    for (i=mpp->xfa_size; i<newsize; ++i) {
      mpp->xbar_forward_addresses[i] = -1;
    }

    mpp->xfa_size = newsize;
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * got an xbar response - create nascent xbar on the port
 * A tagged xbar scout may still be pending, but if it *does* come back, we'll
 * deal with it then.
 */
void
fma_mf_got_xbar_resp(
  struct fma_nic_info *nip,
  int port,
  struct fma_xbar_scout_pkt *pkt)
{
  struct fma_myri *fmp;
  struct fma_map_progress *mpp;
  unsigned char route[FMA_IFC_ROUTE_LEN];
  union lf_node *fnp;
  struct lf_xbar *fxp;
  int route_len;
  int node_index;
  int node_type;
  int node_id;
  int serial;
  int fport;

  fmp = A.myri;
  mpp = fmp->map_progress;
  serial = ntohl(pkt->serial_32);

  /* just return if mapping cancelled */
  if (mpp == NULL || serial != fmp->map_session_id ||
      nip != FMA_NIC(mpp->mf_nic_being_mapped)->local_nip) {
    return;
  }

  /* find the node we were scouting from */
  node_index = ntohl(pkt->node_index_32);
  node_type = LF_TOPO_NI_TYPE(node_index);
  fport = LF_TOPO_NI_PORT(node_index);
  node_id = LF_TOPO_NI_INDEX(node_index);

  /*
   * re-create packet route based on frontier node type
   */
  if (node_type == LF_TOPO_XBAR) {
    fxp = fma_mf_find_xbar_by_node_id(node_id);
    if (fxp == NULL) {
      if (mf_debug) {
	fma_log("Dropping xbar scout of stale xbar x%d", node_id); 
      }
      return;
    }

    fnp = LF_NODE(fxp);

    /* add one byte to route to this xbar */
    route_len = FMA_XBAR(fxp)->mf_route_len+1;
    memcpy(route, FMA_XBAR(fxp)->mf_route, FMA_XBAR(fxp)->mf_route_len);
    route[route_len-1] = LF_DELTA_TO_ROUTE(fport - FMA_XBAR(fxp)->mf_in_port);

  } else {
    fnp = LF_NODE(mpp->mf_nic_being_mapped);

    /* if there is something already here, ignore this pkt */
    if (LF_NIC(fnp)->topo_ports[fport] != NULL) {
      if (mf_debug2) {
	fma_log("NIC %d, p%d already populated", LF_NIC(fnp)->host_nic_id, fport);
      }
      return;
    }
    route_len = 0;
    fxp = NULL;
  }

  if (mf_debug2) {
    fma_log("got return from scout (r=%d), xbar response r:%s",
	mpp->mf_scout_retries_left, lf_route_str(route, route_len));
  }

  /* If frontier is NIC, create the xbar now */
  if (fnp->ln_type == LF_NODE_NIC) {
    fma_mf_create_nascent_xbar(fnp, fport);

  /* otherwise, just leave a marker for now.  We will convert all the markers
   * to real nascent xbars when compares are complete.
   */
  } else {

    /* It's possible something is already here due to merging */
    if (fxp->topo_ports[fport] == NULL) {
      fxp->topo_ports[fport] = FMA_MF_NASCENT_XBAR;

      if (mf_debug2) {
	fma_log("Created nascent placeholder on x%d:p%d",
	  FMA_XBAR(fxp)->node_id, fport);
      }
    } else {
      if (mf_debug2) {
	fma_log("x%d:p%d already populated", FMA_XBAR(fxp)->node_id, fport);
      }
    }
  }
}

/*
 * Create a nascent xbar on the frontier node/port specified
 */
static void
fma_mf_create_nascent_xbar(
  union lf_node *fnp,
  int fport)
{
  struct fma_map_progress *mpp;
  struct lf_xbar *nxp;
  struct fma_nic_info *nip;
  int nport;
  int nic_port;
  unsigned char route[MYRI_MAX_ROUTE_LEN];
  int route_len;

  mpp = A.myri->map_progress;

  /* create nascent xbar */
  nxp = fma_create_xbar(FMA_MF_DFLT_XBAR_PORTS, 0, 0);
  if (nxp == NULL) LF_ERROR(("error reating xbar"));

  fma_mf_increment_node_id();
  FMA_XBAR(nxp)->node_id = mpp->next_node_id;

  if (mf_debug) {
    lf_string_t s;
    int off;

    off = sprintf(s, "created xbar x%d on ", FMA_XBAR(nxp)->node_id);
    if (fnp->ln_type == LF_NODE_NIC) {
      off += sprintf(s+off, "NIC");
    } else {
      off += sprintf(s+off, "x%d", FMA_XBAR_N(fnp)->node_id);
    }
    fma_log("%s:%d", s, fport);
  }

  if (fnp->ln_type == LF_NODE_NIC) {
    nip = FMA_NIC_N(fnp)->local_nip;
    nic_port = fport;
    route_len = 0;

  } else {
    nip = FMA_XBAR_N(fnp)->mf_nic;
    nic_port = FMA_XBAR_N(fnp)->mf_nic_port;
    route_len = FMA_XBAR_N(fnp)->mf_route_len + 1;
    memcpy(route, FMA_XBAR_N(fnp)->mf_route, route_len-1);
    route[route_len-1] = LF_DELTA_TO_ROUTE(fport - FMA_XBAR_N(fnp)->mf_in_port);
  }

  /* record info about this xbar */
  nport = FMA_MF_INITIAL_XBAR_PORT;
  fma_mf_init_xbar_private(nxp, nip, nic_port, route, route_len, nport);

  fma_mf_connect_nodes(fnp, fport, LF_NODE(nxp), nport, FALSE);	/* unresolved */

  /* add to nascent list */
  fma_mf_add_xbar_to_nascent(nxp);
  return;

 except:
  dump_map();
  fma_perror_exit(1);
}

/*
 * Scan through all resolved xbars and convert any lingering nascent xbar
 * placeholders into real xbars.
 */
static void
fma_mf_remove_nascent_placeholders()
{
  struct fma_map_progress *mpp;
  struct fma_xbar *anchor;
  struct fma_xbar *fxp;
  struct lf_xbar *xp;
  int p;

  mpp = A.myri->map_progress;

  anchor = &mpp->resolved_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    xp = fxp->parent;
    for (p=0; p<xp->num_ports; ++p) {
      if (xp->topo_ports[p] == FMA_MF_NASCENT_XBAR) {
	fma_mf_create_nascent_xbar(LF_NODE(xp), p);
      }
    }
  }
}

/*
 * got a tagged xbar response - search for this xid.
 * If it is found, just make the link.  If it is not found, add a
 * new resolved xbar to the map.  If there is already something attached to
 * the frontier port, that means an xbar scout beat us back, and we need to
 * unlink the xbar and reinitialize it.
 */
void
fma_mf_got_tagged_xbar_resp(
  struct fma_nic_info *nip,
  int port,
  struct fma_xbar_scout_pkt *pkt)
{
  struct fma_myri *fmp;
  struct fma_map_progress *mpp;
  unsigned char route[FMA_IFC_ROUTE_LEN];
  int route_len;
  union lf_node *fnp;
  struct lf_xbar *fxp;
  int fport;
  struct lf_xbar *nxp;
  int nport;
  struct lf_xbar *cxp;
  int serial;
  int node_index;
  int node_type;
  int node_id;
  int quad_dis;
  int xbar_size;
  int xbar_ports;
  int cport;
  int xid;
  int rc;

  fmp = A.myri;
  mpp = fmp->map_progress;
  serial = ntohl(pkt->serial_32);

  /* just return if mapping cancelled */
  if (mpp == NULL || serial != fmp->map_session_id ||
      nip != FMA_NIC(mpp->mf_nic_being_mapped)->local_nip) {
    return;
  }

  /* find the node we were scouting from */
  node_index = ntohl(pkt->node_index_32);
  node_type = LF_TOPO_NI_TYPE(node_index);
  fport = LF_TOPO_NI_PORT(node_index);
  node_id = LF_TOPO_NI_INDEX(node_index);

  /*
   * re-create packet route based on frontier node type
   */
  if (node_type == LF_TOPO_XBAR) {
    fxp = fma_mf_find_xbar_by_node_id(node_id);
    if (fxp == NULL) {
      if (mf_debug) {
	fma_log("Dropping tagged xbar scout of stale xbar x%d", node_id); 
      }
      return;
    }

    fnp = LF_NODE(fxp);

    /* add one byte to route to this xbar */
    route_len = FMA_XBAR(fxp)->mf_route_len+1;
    memcpy(route, FMA_XBAR(fxp)->mf_route, FMA_XBAR(fxp)->mf_route_len);
    route[route_len-1] = LF_DELTA_TO_ROUTE(fport - FMA_XBAR(fxp)->mf_in_port);

  } else {
    fnp = LF_NODE(mpp->mf_nic_being_mapped);
    route_len = 0;
    fxp = NULL;
  }

  /* get port and xid of new xbar */
  nport = pkt->tagged_xbar_insert.absolute_port;
  xid = lf_swap_l(ntohl(pkt->tagged_xbar_insert.id));
  quad_dis = pkt->tagged_xbar_insert.quadrant_disabled;
  xbar_size = pkt->tagged_xbar_insert.xbar_size;

  if (mf_debug) {
    lf_string_t s;
    int off;

    off = sprintf(s, "got tagged response, id=%d, p=%d, qd=%d on ",
	xid, nport, quad_dis);
    if (fnp->ln_type == LF_NODE_NIC) {
      off += sprintf(s+off, "NIC");
    } else {
      off += sprintf(s+off, "x%d", FMA_XBAR_N(fnp)->node_id);
    }
    fma_log("%s:%d", s, fport);
  }

  /*
   * See if this xbar is already known to us.  If so, make the link.
   * Also check now to see if there is already something attached to this
   * frontier port.
   */
  nxp = fma_mf_find_xbar_by_xbar_id(xid);
  cxp = LF_XBAR(lf_get_topo_link(fnp, fport, &cport));

  /* If it's just a nascent xbar, destroy it */
  if (cxp == LF_XBAR(FMA_MF_NASCENT_XBAR)) {
    lf_make_topo_link(fnp, fport, NULL, 0);
    cxp = NULL;
  }

  /* if true, this xid has already been seen */
  if (nxp != NULL) {

    if (mf_debug) fma_log("this is x%d", FMA_XBAR(nxp)->node_id);
    
    /* something already attached to this frontier port ? */
    if (cxp != NULL) {

      /* If already attached, nothing to do */
      if (cxp == nxp) {
	if (mf_debug) fma_log("Already attached");

      } else {
	if (cxp->xbar_id != 0) {
	  LF_ERROR(("conflicting tagged xbar connections"));
	}

	/* If this xbar is resolved, subsume it into the tagged xbar */
	if (FMA_XBAR(cxp)->mf_xbar_resolved) {

	  /* merge cxp into nxp and destroy it */
	  if (mf_debug) fma_log("merging in x%d", FMA_XBAR(cxp)->node_id);
	  rc = fma_mf_merge_xbars(cxp, cport, nxp, nport);
	  if (rc == -1) goto restart;

	/* if it's not resolved, just eliminate and connect 
	 * the frontier to this tagged xbar
	 */
	} else {
	  if (mf_debug) fma_log("destroying x%d", FMA_XBAR(cxp)->node_id);
	  fma_mf_destroy_xbar(cxp);
	  fma_mf_connect_nodes(fnp, fport, LF_NODE(nxp), nport, TRUE);
	}
      }

    /* there is nothing attached to our frontier port,
     * is there something attached to the tagged xbar at this spot?  If so, 
     * make sure it is not resolved and then merge it into frontier xbar
     */
    } else {
      struct lf_xbar *ncxp;

      ncxp = LF_XBAR(nxp->topo_ports[nport]);

      /* if nothing real there, a new connection, link it up */
      if (ncxp == NULL || ncxp == LF_XBAR(FMA_MF_NASCENT_XBAR)) {
	fma_mf_connect_nodes(fnp, fport, LF_NODE(nxp), nport, TRUE);

      /* something is there - validate and merge */
      } else {
        int ncport;

	/* XXX(); what if fnp is a NIC ?? */

	if (FMA_XBAR_N(fnp)->mf_xbar_resolved
	    && FMA_XBAR(ncxp)->mf_xbar_resolved) {
	  LF_ERROR(("Unexpected resolved xbar"));
	}

	ncport = nxp->topo_rports[nport];

	/* the merge will hook us to this tagged xbar */
	rc = fma_mf_merge_xbars(fxp, fport, ncxp, ncport);
	if (rc == -1) goto restart;
      }
    }

  /* no matching xid, so this is a new unique xbar */
  } else {

    /* Figure out how big this xbar is */
    xbar_ports = lf_xbar_num_ports(nip->myri_info.speed, xbar_size);

    /* is something already attached to this port ? */
    if (cxp != NULL) {

      /* If this xbar is resolved, convert it to tagged */
      if (FMA_XBAR(cxp)->mf_xbar_resolved) {

	/* create a new tagged xbar and merge this xbar into it */
	nxp = fma_create_xbar(xbar_ports, xid, quad_dis);
	if (nxp == NULL) LF_ERROR(("error creating xbar"));

	fma_mf_increment_node_id();
	FMA_XBAR(nxp)->node_id = mpp->next_node_id;

	if (mf_debug) {
	  fma_log("created tagged xbar%d x%d, merging in x%d", nxp->num_ports,
	      FMA_XBAR(nxp)->node_id, FMA_XBAR(cxp)->node_id);
	}

	/* merge cxp into nxp and destroy it */
	rc = fma_mf_merge_xbars(cxp, cport, nxp, nport);
	if (rc == -1) goto restart;

	fma_mf_add_xbar_to_nascent(nxp); /* leave on nascent list for now */

      /* not resolved, clobber all compares and hook it up as a tagged */
      } else {

	cxp->xbar_id = xid;		/* set xid and qd */
	cxp->quadrant_disable = quad_dis;

	FMA_XBAR(cxp)->mf_xbar_resolved = TRUE;

	if (mf_debug) fma_log("turning unresolved x%d into tagged",
	    FMA_XBAR(cxp)->node_id);

	/* mark these ports closed */
	fma_mf_node_port_closed(LF_NODE(cxp), nport);
	fma_mf_node_port_closed(fnp, fport);
      }

    /* completely new tagged xbar - allocate, fill in,
     * and link into nascent list */
    } else {
      /* create xbar and add it to map */
      nxp = fma_create_xbar(xbar_ports, xid, quad_dis);
      if (nxp == NULL) LF_ERROR(("error reating xbar"));

      fma_mf_increment_node_id();
      FMA_XBAR(nxp)->node_id = mpp->next_node_id;

      if (mf_debug) {
	lf_string_t s;
	int off;

	off = sprintf(s, "created tagged xbar%d x%d on ", nxp->num_ports,
	    FMA_XBAR(nxp)->node_id);
	if (fnp->ln_type == LF_NODE_NIC) {
	  off += sprintf(s+off, "NIC");
	} else {
	  off += sprintf(s+off, "x%d", FMA_XBAR_N(fnp)->node_id);
	}
	fma_log("%s:%d", s, fport);
      }

      /* record info about this xbar */
      fma_mf_init_xbar_private(nxp, nip, port, route, route_len, nport);

      fma_mf_add_xbar_to_nascent(nxp);    /* add this xbar to nascent */

      /* link to this node */
      fma_mf_connect_nodes(fnp, fport, LF_NODE(nxp), nport, TRUE);

      /* tagged xbars are resolved automatically */
      FMA_XBAR(nxp)->mf_xbar_resolved = TRUE;
    }
  }
  return;

 restart:
  fma_mf_restart_mapping();
  return;

 except:
  dump_map();
  fma_perror_exit(1);
}

/*
 * Stop any compares in progress on this xbar
 */
static void
fma_mf_stop_compares(
  struct lf_xbar *xp)
{
  struct fma_map_progress *mpp;
  struct fma_xbar *fxp;

  mpp = A.myri->map_progress;
  fxp = FMA_XBAR(xp);

  /* no longer comparing */
  mpp->mf_outstanding_scouts -= fxp->mf_compares_sent;
  fxp->mf_compares_sent = 0;

  /*
   * Clear compare timeout timer if active
   */
  if (fxp->mf_compare_timer != NULL) {
    fma_sync_timer_cancel(fxp->mf_compare_timer);
    fxp->mf_compare_timer = NULL;
  }
}

/*
 * connect two nodes together and mark the link as resolved
 */
static void
fma_mf_connect_nodes(
  union lf_node *np1,
  int port1,
  union lf_node *np2,
  int port2,
  int resolved)
{
  /* link the ports */
  lf_make_topo_link(np1, port1, np2, port2);
  lf_make_topo_link(np2, port2, np1, port1);

  /* mark both ports as resolved */
  if (resolved) {
    fma_mf_node_port_closed(np1, port1);
    fma_mf_node_port_closed(np2, port2);
  }
}

/*
 * Explore all xbars on the frontier for NICs
 */
static void
fma_mf_explore_frontier_for_nics()
{
  struct fma_map_progress *mpp;
  struct fma_settings *asp;

  asp = A.settings;
  mpp = A.myri->map_progress;

  /* just set retry count and use the timeout routine to start things */
  mpp->mf_scout_retries_left = asp->nic_scout_retries + 1;
  fma_mf_nic_scout_timeout(NULL);
}

/*
 * NIC scouting has timed out.  If retries left, try resending.
 * If no retries, or nothing sent, then move on to xbar scouts.
 */
static void
fma_mf_nic_scout_timeout(
  void *v)
{
  struct fma_map_progress *mpp;
  struct fma_xbar *frontier;
  struct fma_settings *asp;
  struct fma_xbar *fxp;
  struct lf_nic *nicp;
  int port;

  asp = A.settings;
  mpp = A.myri->map_progress;
  nicp = mpp->mf_nic_being_mapped;
  port = mpp->mf_port_being_mapped;

  mpp->mf_scout_timer = NULL;		/* clear timer ptr */
  mpp->mf_outstanding_scouts = 0;	/* no scouts outstanding */

  /* If any retries left, re-send the scout */
  if (mpp->mf_scout_retries_left > 0) {
    --mpp->mf_scout_retries_left;

    /* explore every xbar on the frontier */
    frontier = &mpp->frontier_xbar_anchor;
    for (fxp=frontier->mf_next; fxp != frontier; fxp=fxp->mf_next) {
      fma_mf_explore_xbar_for_nics(fxp->parent);
    }

    /* If any scouts sent, then start a timer to await them */
    if (mpp->mf_outstanding_scouts > 0) {

      /* start a sync timer for the NIC scouts */
      mpp->mf_scout_timer = fma_sync_timer_setup(
		FMA_NIC(nicp)->local_nip->nic_handle, port,
		fma_mf_nic_scout_timeout, NULL, asp->nic_scout_timeout);
      if (mpp->mf_scout_timer == NULL) {
	LF_ERROR(("Error setting up NIC scout timer"));
      }
    } else {

      /* If nothing got sent, move on to next phase */
      fma_mf_exploration_done();
    }

  /* If no retries left, proceed to scouting for a xbars */
  } else {

    /* just set retry count and use the timeout routine to start things */
    mpp->mf_scout_retries_left = FMA_XBAR_SCOUT_RETRY+1;
    fma_mf_xbar_scout_timeout(NULL);
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * xbar scouting has timed out.  If retries left, try resending.
 * If no retries, or nothing sent, then move on to xbar scouts.
 */
static void
fma_mf_xbar_scout_timeout(
  void *v)
{
  struct fma_map_progress *mpp;
  struct fma_xbar *frontier;
  struct fma_xbar *fxp;
  struct lf_nic *nicp;
  int port;

  mpp = A.myri->map_progress;
  nicp = mpp->mf_nic_being_mapped;
  port = mpp->mf_port_being_mapped;

  mpp->mf_scout_timer = NULL;		/* clear timer ptr */
  mpp->mf_outstanding_scouts = 0;	/* no scouts outstanding */

  /* If any retries left, re-send the scout */
  if (mpp->mf_scout_retries_left > 0) {
    --mpp->mf_scout_retries_left;

    /* explore every xbar on the frontier */
    frontier = &mpp->frontier_xbar_anchor;
    for (fxp=frontier->mf_next; fxp != frontier; fxp=fxp->mf_next) {
      fma_mf_explore_xbar_for_xbars(fxp->parent);
    }

    /* If any scouts sent, then start a timer to await them */
    if (mpp->mf_outstanding_scouts > 0) {

      /* start a sync timer for the xbar scout */
      mpp->mf_scout_timer = fma_sync_timer_setup(
		FMA_NIC(nicp)->local_nip->nic_handle, port,
		fma_mf_xbar_scout_timeout, NULL, FMA_XBAR_SCOUT_TIMEOUT);
      if (mpp->mf_scout_timer == NULL) {
	LF_ERROR(("Error setting up xbar scout timer"));
      }
    }
  }
  
  /* If nothing got sent (either because timeout or nothing to send),
   * move on to next phase
   */
  if (mpp->mf_outstanding_scouts <= 0) {
    fma_mf_exploration_done();
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Explore all the ports on this xbar for NICs
 */
static void
fma_mf_explore_xbar_for_nics(
  struct lf_xbar *xp)
{
  int p;
  struct fma_mf_info *mfip;

  /* explore every open port */
  for (p=0; p<xp->num_ports; ++p) {

    mfip = FMA_XBAR(xp)->mf_info + p;

    if (xp->topo_ports[p] == NULL && !mfip->mf_closed) {
      fma_mf_scout_node_for_nics(FMA_XBAR(xp)->mf_nic,
	  FMA_XBAR(xp)->mf_nic_port, LF_NODE(xp), p);
    }
  }
}

/*
 * Explore all the ports on this xbar for other xbars
 */
static void
fma_mf_explore_xbar_for_xbars(
  struct lf_xbar *xp)
{
  int p;
  struct fma_mf_info *mfip;

  /* explore every open port */
  for (p=0; p<xp->num_ports; ++p) {

    mfip = FMA_XBAR(xp)->mf_info + p;

    if (!mfip->mf_closed) {
      fma_mf_scout_node_for_xbars(FMA_XBAR(xp)->mf_nic,
	  FMA_XBAR(xp)->mf_nic_port, LF_NODE(xp), p);
    }
  }
}

/*
 * Start xbar compares on this xbar, which is on the frontier list
 */
static int
fma_mf_start_xbar_compares(
  struct lf_xbar *xp,
  int port)
{
  struct fma_map_progress *mpp;
  int sent;

  mpp = A.myri->map_progress;

  /* Send all the compares for this xbar */
  sent = fma_mf_send_xbar_compares(xp, port);
  FMA_XBAR(xp)->mf_compares_sent = sent;

  /* If compares in progress, note this and start timer */
  if (sent > 0) {

    /* Keep track of total number of scouts sent */
    mpp->mf_outstanding_scouts += FMA_XBAR(xp)->mf_compares_sent;

    /* set up retry count */
    FMA_XBAR(xp)->mf_compare_port = port;
    ++FMA_XBAR(xp)->mf_compare_tries;

    /* start timer when sends complete */
    FMA_XBAR(xp)->mf_compare_timer = fma_sync_timer_setup(
		FMA_XBAR(xp)->mf_nic->nic_handle, FMA_XBAR(xp)->mf_nic_port,
		fma_mf_compare_timeout, xp, FMA_XBAR(xp)->mf_compare_timeout);

  } else {
    /* XXX - let caller do this */

    if (mf_debug) fma_log("No compares sent for x%d, unique",
	FMA_XBAR(xp)->node_id);

    fma_mf_xbar_is_unique(xp);
  }
  return sent;
}

static int
fma_mf_send_xbar_compares(
  struct lf_xbar *xp,
  int port)
{
  int sent;

  sent = 0;

  if (FMA_XBAR(xp)->mf_compare_stage < 1) {
    sent = fma_mf_send_stage0_compares(xp, port);
    ++FMA_XBAR(xp)->mf_compare_stage;
    FMA_XBAR(xp)->mf_compare_timeout = 8;
  }

  if (FMA_XBAR(xp)->mf_compare_stage < 2 && sent == 0) {
    sent = fma_mf_send_stage1_compares(xp, port);
    ++FMA_XBAR(xp)->mf_compare_stage;
    FMA_XBAR(xp)->mf_compare_timeout = 8;
  }

  if (sent == 0) {
    sent = fma_mf_send_stage2_compares(xp, port);
    FMA_XBAR(xp)->mf_compare_timeout = 20;
  }

  return sent;
}

static int
fma_mf_send_stage0_compares(
  struct lf_xbar *xp,
  int port)
{
  struct fma_map_progress *mpp;
  struct fma_mf_info *mfip;
  struct fma_xbar *anchor;
  struct fma_xbar *ftxp;
  struct fma_xbar *fuxp;
  struct lf_xbar *txp;
  int sent;

  mpp = A.myri->map_progress;
  sent = 0;
  fuxp = FMA_XBAR(xp);

  /* loop through all resolved xbars for possible compares */
  anchor = &mpp->resolved_xbar_anchor;
  for (ftxp=anchor->mf_next; ftxp != anchor; ftxp=ftxp->mf_next) {
    txp = ftxp->parent;

    /* skip if signatures do not match */
    if (ftxp->mf_signature != FMA_XBAR(xp)->mf_signature) continue;

    /* make last route byte match */
    if (ftxp->mf_route_len != fuxp->mf_route_len ||
	ftxp->mf_route[ftxp->mf_route_len-1] 
	!= fuxp->mf_route[fuxp->mf_route_len-1]) {
      continue;
    }

    mfip = ftxp->mf_info + port;

    /* send compare to port if open */
    if (!mfip->mf_closed) {
      fma_mf_compare_xbar_port(xp, port, txp);
      ++sent;
    }
  }

  /* loop through all frontier xbars for possible compares */
  anchor = &mpp->frontier_xbar_anchor;
  for (ftxp=anchor->mf_next; ftxp != anchor; ftxp=ftxp->mf_next) {
    txp = ftxp->parent;

    if (txp == xp) continue;	/* don't compare to self */

    /* skip if signatures do not match */
    if (ftxp->mf_signature != FMA_XBAR(xp)->mf_signature) continue;

    /* make last route byte match */
    if (ftxp->mf_route_len != fuxp->mf_route_len ||
	ftxp->mf_route[ftxp->mf_route_len-1] 
	!= fuxp->mf_route[fuxp->mf_route_len-1]) {
      continue;
    }

    mfip = ftxp->mf_info + port;

    /* send compare to port if open */
    if (!mfip->mf_closed) {
      fma_mf_compare_xbar_port(xp, port, txp);
      ++sent;
    }
  }

  /* loop through all resolving xbars for possible compares */
  anchor = &mpp->resolving_xbar_anchor;
  for (ftxp=anchor->mf_next; ftxp != anchor; ftxp=ftxp->mf_next) {
    txp = ftxp->parent;

    if (txp == xp) continue;	/* don't compare to self */

    /* skip if signatures do not match */
    if (ftxp->mf_signature != FMA_XBAR(xp)->mf_signature) continue;

    /* make last route byte match */
    if (ftxp->mf_route_len != fuxp->mf_route_len ||
	ftxp->mf_route[ftxp->mf_route_len-1] 
	!= fuxp->mf_route[fuxp->mf_route_len-1]) {
      continue;
    }

    mfip = ftxp->mf_info + port;

    /* send compare to port if open */
    if (!mfip->mf_closed) {
      fma_mf_compare_xbar_port(xp, port, txp);
      ++sent;
    }
  }

  if (mf_debug) fma_log("sent %d stage0 compares for x%d:p%d", sent,
			FMA_XBAR(xp)->node_id, port);

  return sent;
}

static int
fma_mf_send_stage1_compares(
  struct lf_xbar *xp,
  int port)
{
  return 0;
}

static int
fma_mf_send_stage2_compares(
  struct lf_xbar *xp,
  int port)
{
  struct fma_map_progress *mpp;
  struct fma_mf_info *mfip;
  struct fma_xbar *anchor;
  struct fma_xbar *ftxp;
  struct lf_xbar *txp;
  int sent;

  mpp = A.myri->map_progress;
  sent = 0;

  /* loop through all resolved xbars for possible compares */
  anchor = &mpp->resolved_xbar_anchor;
  for (ftxp=anchor->mf_next; ftxp != anchor; ftxp=ftxp->mf_next) {
    txp = ftxp->parent;

    /* skip if signatures do not match */
    if (ftxp->mf_signature != FMA_XBAR(xp)->mf_signature) continue;

    mfip = ftxp->mf_info + port;

    /* send compare to port if open */
    if (!mfip->mf_closed) {
      fma_mf_compare_xbar_port(xp, port, txp);
      ++sent;
    }
  }

  /* loop through all frontier xbars for possible compares */
  anchor = &mpp->frontier_xbar_anchor;
  for (ftxp=anchor->mf_next; ftxp != anchor; ftxp=ftxp->mf_next) {
    txp = ftxp->parent;

    if (txp == xp) continue;	/* don't compare to self */

    /* skip if signatures do not match */
    if (ftxp->mf_signature != FMA_XBAR(xp)->mf_signature) continue;

    /* XXX don't compare to farther away than us */
    if (ftxp->mf_route_len > FMA_XBAR(xp)->mf_route_len) continue;

    mfip = ftxp->mf_info + port;

    /* send compare to port if open */
    if (!mfip->mf_closed) {
      fma_mf_compare_xbar_port(xp, port, txp);
      ++sent;
    }
  }

  /* loop through all resolving xbars for possible compares */
  anchor = &mpp->resolving_xbar_anchor;
  for (ftxp=anchor->mf_next; ftxp != anchor; ftxp=ftxp->mf_next) {
    txp = ftxp->parent;

    if (txp == xp) continue;	/* don't compare to self */

    /* skip if signatures do not match */
    if (ftxp->mf_signature != FMA_XBAR(xp)->mf_signature) continue;

    /* XXX don't compare to farther away than us */
    if (ftxp->mf_route_len > FMA_XBAR(xp)->mf_route_len) continue;

    mfip = ftxp->mf_info + port;

    /* send compare to port if open */
    if (!mfip->mf_closed) {
      fma_mf_compare_xbar_port(xp, port, txp);
      ++sent;
    }
  }

  if (mf_debug) fma_log("sent %d stage2 compares for x%d:p%d", sent,
			FMA_XBAR(xp)->node_id, port);

  return sent;
}

/*
 * Send an xbar compare to test whether we are this port
 */
static void
fma_mf_compare_xbar_port(
  struct lf_xbar *xp,
  int port,
  struct lf_xbar *txp)
{
  struct fma_xbar_compare_pkt pkt;
  struct fma_xbar_compare_pkt *compp;
  struct fma_map_progress *mpp;
  unsigned char route[MYRI_MAX_ROUTE_LEN];
  int route_len;
  int rc;

  mpp = A.myri->map_progress;

  /* Fill in the packet */
  compp = &pkt;
  compp->hdr.type_16 = htons(FMA_PACKET_TYPE);
  compp->hdr.subtype_16 = htons(FMA_SUBTYPE_XBAR_COMPARE);
  compp->map_session_id_32 = htonl(A.myri->map_session_id);
  compp->unres_node_id_32 = htonl(FMA_XBAR(xp)->node_id);
  compp->unres_port_8 = port;
  compp->test_node_id_32 = htonl(FMA_XBAR(txp)->node_id);

  /* save packet origin */
  LF_MAC_COPY(compp->origin_mac_addr, FMA_XBAR(xp)->mf_nic->myri_info.mac_addr);
  compp->origin_port_8 = FMA_XBAR(xp)->mf_nic_port;

  /*
   * create route that would be loop if xp:port == txp:port.  We will send a
   * route that will go out to xp, and come back along the txp route inverted.
   */
  memcpy(route, FMA_XBAR(txp)->mf_route, FMA_XBAR(txp)->mf_route_len);
  route[FMA_XBAR(txp)->mf_route_len] =
       LF_DELTA_TO_ROUTE(port - FMA_XBAR(txp)->mf_in_port);
  lf_reverse_route(route + FMA_XBAR(txp)->mf_route_len + 1,
                   FMA_XBAR(xp)->mf_route, FMA_XBAR(xp)->mf_route_len);
  route_len = FMA_XBAR(xp)->mf_route_len + FMA_XBAR(txp)->mf_route_len + 1;

#if 0
  if (mf_debug) fma_log("compare x%d:p%d to x%d:p%d, rt = %s", FMA_XBAR(xp)->node_id, port, FMA_XBAR(txp)->node_id, port, lf_route_str(route, route_len));
#endif

  /* send the compare packet */
  rc = fma_myri_raw_send(FMA_XBAR(xp)->mf_nic->nic_handle,
                         FMA_XBAR(xp)->mf_nic_port,
			 route, route_len,
      			 &pkt, sizeof(pkt), NULL, NULL);
  if (rc != 0) LF_ERROR(("Error sending xbar compare packet"));

  return;

 except:
  dump_map();
  fma_perror_exit(1);
}

/*
 * xbar comparisons timed out, retry if needed, else this xbar is unique.
 */
static void
fma_mf_compare_timeout(
  void *vxp)
{
  struct fma_map_progress *mpp;
  struct lf_xbar *xp;

  xp = vxp;
  mpp = A.myri->map_progress;
  FMA_XBAR(xp)->mf_compare_timer = NULL;

  if (mf_debug2) fma_log("compare timer popped for x%d",
      FMA_XBAR(xp)->node_id);

  mpp->mf_outstanding_scouts -= FMA_XBAR(xp)->mf_compares_sent;
  FMA_XBAR(xp)->mf_compares_sent = 0;

  /*
   * If retries left, re-send the packets
   */
  if (FMA_XBAR(xp)->mf_compare_tries <= FMA_XBAR_SCOUT_RETRY) {
    
    /* Put this back on frontier, we'll get back to it later */
    fma_mf_unlink_xbar_from_mf_list(xp);
    fma_mf_add_xbar_to_frontier(xp);
      
  /* Out of retries --> This xbar is unique */
  } else {
    fma_mf_xbar_is_unique(xp);
  }

  /* see if there are any more xbar compares to start */
  fma_mf_start_next_xbar_compares();
}

/*
 * Declare an unresolved xbar unique
 */
static void
fma_mf_xbar_is_unique(
  struct lf_xbar *xp)
{
  int port;

  /* mark both ends of any links to NICs or resolved xbars as resolved */
  for (port=0; port<xp->num_ports; ++port) {
    union lf_node *onp;
    int oport;

    onp = xp->topo_ports[port];
    if (onp == NULL || onp == FMA_MF_NASCENT_XBAR) continue;

    oport = xp->topo_rports[port];

    if (onp->ln_type == LF_NODE_NIC
        || (onp->ln_type == LF_NODE_XBAR
	    && FMA_XBAR_N(onp)->mf_xbar_resolved)) {
      fma_mf_node_port_closed(LF_NODE(xp), port);
      fma_mf_node_port_closed(onp, oport);
    }
  }
  
  /* mark this xbar as resolved */
  FMA_XBAR(xp)->mf_xbar_resolved = TRUE;

  /* move the xbar from wherever is was to resolved */
  fma_mf_unlink_xbar_from_mf_list(xp);
  fma_mf_add_xbar_to_resolved(xp);

  if (mf_debug) fma_log("xbar x%d is unique", FMA_XBAR(xp)->node_id);
}

/*
 * Unlink and destroy this xbar
 */
static void
fma_mf_destroy_xbar(
  struct lf_xbar *xp)
{
  /* Stop any compares from this xbar */
  fma_mf_stop_compares(xp);

  fma_mf_unlink_xbar_from_mf_list(xp);	/* unlink the xbar */
  lf_free_xbar(xp);
}

/*
 * Find an xbar in the map by its xbar ID
 * Look everywhere except for "resolving"
 */
static struct lf_xbar *
fma_mf_find_xbar_by_xbar_id(
  int xbar_id)
{
  struct fma_xbar *anchor;
  struct fma_xbar *fxp;

  anchor = &A.myri->map_progress->resolved_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    if (fxp->parent->xbar_id == xbar_id) {
      return fxp->parent;
    }
  }

  anchor = &A.myri->map_progress->frontier_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    if (fxp->parent->xbar_id == xbar_id) {
      return fxp->parent;
    }
  }

  anchor = &A.myri->map_progress->nascent_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    if (fxp->parent->xbar_id == xbar_id) {
      return fxp->parent;
    }
  }
  return NULL;
}

/*
 * Find an xbar in the map by its node ID - look everywhere
 */
static struct lf_xbar *
fma_mf_find_xbar_by_node_id(
  int node_id)
{
  struct fma_xbar *anchor;
  struct fma_xbar *fxp;

  anchor = &A.myri->map_progress->frontier_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    if (fxp->node_id == node_id) {
      return fxp->parent;
    }
  }

  anchor = &A.myri->map_progress->resolving_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    if (fxp->node_id == node_id) {
      return fxp->parent;
    }
  }

  anchor = &A.myri->map_progress->resolved_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    if (fxp->node_id == node_id) {
      return fxp->parent;
    }
  }

  anchor = &A.myri->map_progress->nascent_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    if (fxp->node_id == node_id) {
      return fxp->parent;
    }
  }
  return NULL;
}

/*
 * Unlink an xbar from an mf_next based list
 */
static void
fma_mf_unlink_xbar_from_mf_list(
  struct lf_xbar *xp)
{
  FMA_XBAR(xp)->mf_prev->mf_next = FMA_XBAR(xp)->mf_next;
  FMA_XBAR(xp)->mf_next->mf_prev = FMA_XBAR(xp)->mf_prev;
}

/*
 * Following forwarding chain.
 */
static void
fma_mf_calculate_forwarding(
  int *idp,
  int *port)
{
  struct fma_map_progress *mpp;
  int id;
  int nid;
  lf_string_t s;
  int delta;
  int off;

  mpp = A.myri->map_progress;
  off = 0;
  delta = 0;

  nid = *idp;
  do {
    id = nid;
    nid = mpp->xbar_forward_addresses[id];
    if (nid != -1) {
      delta += mpp->xbar_forward_deltas[id];
    }

    if (mf_debug && nid != -1) {
      off += sprintf(s+off, "%s x%d -> x%d(%s%d)", 
	(id == *idp) ? "fwd" : ",",
	id, nid, 
	(delta<0)?"":"+",delta);
    }

  } while (nid != -1);

  if (mf_debug && (id != *idp)) fma_log(s);

  *idp = id;
  *port += delta;
}

/*
 * Got a response from an xbar compare packet.  This means we found the
 * crossbar under test to be a duplicate of an existing one.  Merge them.
 */
void
fma_mf_got_compare_resp(
  struct fma_nic_info *nip,
  int port,
  struct fma_xbar_compare_pkt *compp)
{
  int unid;
  struct lf_xbar *uxp;
  int uport;
  int tnid;
  struct lf_xbar *txp;
  int tport;
  union lf_node *cnp;
  int rc;

  /* Drop this if not mapping or not for this session */
  if (A.myri->map_progress == NULL
      || ntohl(compp->map_session_id_32) != A.myri->map_session_id
      || LF_MAC_CMP(compp->origin_mac_addr, nip->myri_info.mac_addr) != 0
      || compp->origin_port_8 != port) {
    return;
  }

  /* pull data from the packet */
  unid = ntohl(compp->unres_node_id_32);
  uport = compp->unres_port_8;
  tnid = ntohl(compp->test_node_id_32);
  tport = uport;			/* ports are always the same now */

  if (mf_debug) {
    fma_log("unres x%d:p%d matches x%d:p%d", unid, uport, tnid, tport);
  }

  /* see if these guys moved and left forwarding address */
  fma_mf_calculate_forwarding(&unid, &uport);
  fma_mf_calculate_forwarding(&tnid, &tport);

  /* If IDs now equal, nothing left to do! */
  if (unid == tnid) {
    if (mf_debug) fma_log("Already merged!");
    return;
  }

  uxp = fma_mf_find_xbar_by_node_id(unid);
  if (uxp == NULL) {
    if (mf_debug) fma_log("stale: unres x%d not found", unid);
    return;
  }

  /* 
   * If compare is not active, this set of compares got cancelled
   * or something.
   */
  if (FMA_XBAR(uxp)->mf_compares_sent == 0) {
    if (mf_debug) fma_log("stale: unres x%d not comparing", unid);
    return;
  }

  txp = fma_mf_find_xbar_by_node_id(tnid);
  if (txp == NULL) {
    if (mf_debug) fma_log("stale: test x%d not found", tnid);
    return;
  }

  /* If connection if just a nascent placeholder, clobber it */
  cnp = txp->topo_ports[tport];
  if (cnp == FMA_MF_NASCENT_XBAR) {
    txp->topo_ports[tport] = NULL;

  /* is this other port already connected to something? */
  } else if (cnp != NULL) {
    struct lf_xbar *cxp;

    /* make sure it's at least an xbar */
    if (cnp->ln_type != LF_NODE_XBAR) {
      LF_ERROR(("Bad node type collides with compare"));
    }

    cxp = LF_XBAR(cnp);
    if (FMA_XBAR(cxp)->mf_xbar_resolved) {
      LF_ERROR(("Resolved xbar collides with compare"));
    }

    /* If we find ourself here, this is a loopback connection,
     * don't destroy the xbar
     */
    if (cxp != uxp) {

      /* now we know we have a ghost xbar at this spot - nuke it */
      txp->topo_ports[tport] = NULL;
      fma_mf_destroy_xbar(cxp);
    } else {
      if (mf_debug2) {
	fma_log("loopback discovered?");
      }
    }
  }

  if (mf_debug2) {
    fma_log("  MM uxp, p%d i%d d%d r%d rt: %s", uport,
	FMA_XBAR(uxp)->mf_in_port,
	FMA_XBAR(uxp)->mf_route_len,
	FMA_XBAR(uxp)->mf_xbar_resolved,
	lf_route_str(FMA_XBAR(uxp)->mf_route, FMA_XBAR(uxp)->mf_route_len));
    fma_log("  MM txp, p%d i%d d%d r%d rt: %s", tport,
	FMA_XBAR(txp)->mf_in_port,
	FMA_XBAR(txp)->mf_route_len,
	FMA_XBAR(txp)->mf_xbar_resolved,
	lf_route_str(FMA_XBAR(txp)->mf_route, FMA_XBAR(txp)->mf_route_len));
    fma_log("  MM");
  }

  /* Merge all port connections */
  if (FMA_XBAR(txp)->mf_xbar_resolved) {

    /* Merge this xbar into txp.  this merge destroys uxp and cancels
     * any compare in progress on uxp
     */
    rc = fma_mf_merge_xbars(uxp, uport, txp, tport);
    if (rc == -1) goto except;

  } else {
    rc = fma_mf_merge_xbars(txp, tport, uxp, uport);
    if (rc == -1) goto except;
  }

  /* start up the next set of compares */
  fma_mf_start_next_xbar_compares();
  return;

 except:
  fma_perror();
  fma_mf_restart_mapping();
  return;
}

/*
 * Start xbar compares on the tail xbar of frontier list, if there is one
 */
static void
fma_mf_start_next_xbar_compares()
{
  struct fma_xbar *fxp;
  struct fma_xbar *anchor;
  struct fma_map_progress *mpp;
  int sent;

  mpp = A.myri->map_progress;

  anchor = &mpp->frontier_xbar_anchor;

  while (mpp->mf_outstanding_scouts < FMA_MF_MAX_COMPARE_PROBES) {

    /* If list is not empty, start a compare on the tail */
    fxp = anchor->mf_prev;
    if (fxp != anchor) {
      struct lf_xbar *xp;

      xp = fxp->parent;

      /* already resolved? */
      if (fxp->mf_xbar_resolved) {
	if (mf_debug) fma_log("x%d is already resolved", fxp->node_id);
	fma_mf_xbar_is_unique(xp);

      } else {
	sent = fma_mf_start_xbar_compares(fxp->parent, fxp->mf_in_port);

	/* move the xbar from frontier to resolving */
	if (sent > 0) {
	  fma_mf_unlink_xbar_from_mf_list(xp);
	  fma_mf_add_xbar_to_resolving(xp);
	}
      }

    } else {	/* list empty - nothing to do */
      break;
    }
  }

  /*
   * If no outstanding scouts, all compares are now done!
   */
  if (mpp->mf_outstanding_scouts <= 0) {
    fma_mf_done_comparing();
  }
}

/*
 * All done with this round of comparisons - both frontier and unresolved
 * should now be empty.  Take everything from the nascent list and move it
 * to the frontier for exploration.
 */
static void
fma_mf_done_comparing()
{
  struct fma_map_progress *mpp;
  struct fma_xbar *nascent;
  struct fma_xbar *frontier;

  mpp = A.myri->map_progress;

  if (mf_debug) fma_log("Done with compare phase");

  /* convert all nascent placeholders into real xbars */
  fma_mf_remove_nascent_placeholders();

  /* Move the nascent list to the frontier list in one shot */
  nascent = &mpp->nascent_xbar_anchor;
  frontier = &mpp->frontier_xbar_anchor;

  /* only move if there is something to move */
  if (nascent->mf_next != nascent) {

    /* move to frontier */
    frontier->mf_next = nascent->mf_next;
    frontier->mf_prev = nascent->mf_prev;
    frontier->mf_next->mf_prev = frontier;
    frontier->mf_prev->mf_next = frontier;
    
    /* empty nascent */
    nascent->mf_next = nascent;
    nascent->mf_prev = nascent;

    /* start exploring the frontier */
    fma_mf_explore_frontier_for_nics();

  /* If no nascent xbars, we are done with this NIC */
  } else {
    fma_start_mapping_next_nic_port();
  }
}

/*
 * Merge all xp1 links into xp2 and destroy xp1
 */
static int
fma_mf_merge_xbars(
  struct lf_xbar *xp1,
  int port1,
  struct lf_xbar *xp2,
  int port2)
{
  struct fma_map_progress *mpp;
  int diff;
  int p1;
  int p2;

  mpp = A.myri->map_progress;
  diff = port2 - port1;

  if (mf_debug) fma_log("merge x%d into x%d, diff=%d",
		      FMA_XBAR(xp1)->node_id, FMA_XBAR(xp2)->node_id, diff);

  for (p1=0; p1<xp1->num_ports; ++p1) {
    union lf_node *onp1;
    union lf_node *onp2;
    int oport;

    /* if nothing on xp1's port, skip */
    onp1 = xp1->topo_ports[p1];
    if (onp1 == NULL) continue;

    /* get xp2's connection */
    p2 = p1 + diff;
    if (p2 < 0 || p2 >= xp2->num_ports) {
      LF_ERROR(("Impossible port %d in xbar", p2));
    }
    onp2 = xp2->topo_ports[p2];

    /*
     * Deal with any nascent placeholders here
     */
    if (onp1 == FMA_MF_NASCENT_XBAR) {
      xp1->topo_ports[p1] = NULL;
      if (onp2 == NULL) {
	xp2->topo_ports[p2] = FMA_MF_NASCENT_XBAR;
      }
      continue;
    } else if (onp2 == FMA_MF_NASCENT_XBAR) {
      onp2 = NULL;
      xp2->topo_ports[p2] = NULL;
    }

    /* If we see a NIC here, it's just because of vagaries of the order
     * of responses.  Just disconnect it since the scouts for xp2 will pick
     * it up soon enough.
     */
    if (onp1->ln_type != LF_NODE_XBAR) {
      if (onp2 != NULL) {
	LF_ERROR(("Merging xbar with conflicting NIC attached!"));
      }

      /* break this link */
      LF_NIC(onp1)->topo_ports[xp1->topo_rports[p1]] = NULL;
      xp1->topo_ports[p1] = NULL;

    /*
     * At least one of onp1 and onp1 should be unresolved.  Remove the
     * unresolved one or onp1 if both are.
     */
    } else {

      /* onp1 == xp2 means loopback discovered - don't destroy anything */
      if (LF_XBAR(onp1) == xp2) {
	xp1->topo_ports[p1] = NULL;
	fma_mf_connect_nodes(LF_NODE(xp2), p2,
	       LF_NODE(xp2), xp1->topo_rports[p1],
	       FMA_XBAR(xp2)->mf_xbar_resolved);

      /* resolved xbar on 2 */ 
      } else if (onp2 != NULL && FMA_XBAR_N(onp2)->mf_xbar_resolved) {

	/* if resolved xbar on 1, problem... */
	if (FMA_XBAR_N(onp1)->mf_xbar_resolved) {
	  LF_ERROR(("Both xbars resolved??"));

	/* destroy onp1 */
	} else {
	  xp1->topo_ports[p1] = NULL;
	  LF_XBAR(onp1)->topo_ports[xp1->topo_rports[p1]] = NULL;
	  fma_mf_destroy_xbar(LF_XBAR(onp1));
	}

      /* 2 is NULL or not resolved - if 1 is resolved, * destroy 2,
       * else destroy 1
       */
      } else {

	/* destroying onp2 - need to move from xp1 */
	if (FMA_XBAR_N(onp1)->mf_xbar_resolved) {

	  if (onp2 != NULL) {
	    xp2->topo_ports[p2] = NULL;		/* disconnect for safety */
	    LF_XBAR(onp2)->topo_ports[xp2->topo_rports[p2]] = NULL;
	    fma_mf_destroy_xbar(LF_XBAR(onp2));
	  }

	  /* connect xp2 to onp1 */
	  oport = xp1->topo_rports[p1];
      
	  if (mf_debug) fma_log("  connect x%d:p%d to x%d:p%d",
		FMA_XBAR(xp2)->node_id, p2, FMA_XBAR_N(onp1)->node_id, oport);

	  fma_mf_connect_nodes(LF_NODE(xp2), p2, onp1, oport,
			       FMA_XBAR(xp2)->mf_xbar_resolved);
	  xp1->topo_ports[p1] = NULL;		/* disconnect for safety */

	/* disconnect and destroy onp1 */
	} else {
	  xp1->topo_ports[p1] = NULL;
	  LF_XBAR(onp1)->topo_ports[xp1->topo_rports[p1]] = NULL;
	  fma_mf_destroy_xbar(LF_XBAR(onp1));
	}
      }
    }
  }

  /* save a forwarding address in case a comparison involving this xbar
   * returns
   */
  mpp->xbar_forward_addresses[FMA_XBAR(xp1)->node_id] = FMA_XBAR(xp2)->node_id;
  mpp->xbar_forward_deltas[FMA_XBAR(xp1)->node_id] = diff;

  fma_mf_destroy_xbar(xp1);	/* destroy the old xbar */
  return 0;

 except:
  if (mf_debug) {
    fma_perror();
  } else {
    fma_reset_error();
  }
  return -1;
}

static void
dump_xbar_list(
  struct fma_xbar *anchor,
  char *string)
{
  union lf_node *onp;
  struct fma_xbar *fxp;
  lf_string_t s;
  int off;

  fma_log("[%s list]", string);

  /* loop through all xbars in the map */
  for (fxp=anchor->mf_next; fxp != anchor; fxp=fxp->mf_next) {
    struct lf_xbar *xp;
    int p;

    xp = fxp->parent;
    fma_log("x%d: (xid=%d, qd=%d)", fxp->node_id, xp->xbar_id,
	xp->quadrant_disable);

    for (p=0; p<xp->num_ports; ++p) {
      struct fma_mf_info *mfip;

      off = sprintf(s, "%d ", p);

      mfip = fxp->mf_info + p;
      if (mfip->mf_closed) off += sprintf(s+off, "[closed] ");

      onp = xp->topo_ports[p];
      if (onp == NULL) {

	off += sprintf(s+off, "disconnected");

      } else if (onp == FMA_MF_NASCENT_XBAR) {
	off += sprintf(s+off, "NASCENT_XBAR");

      } else if (onp->ln_type == LF_NODE_NIC) {
	struct lf_nic *nicp;

	nicp = LF_NIC(onp);
	off += sprintf(s+off, " %s NIC %d mac=" LF_MAC_FORMAT ", p%d ",
	    	       nicp->host->hostname, nicp->host_nic_id, 
		       LF_MAC_ARGS(nicp->mac_addr), xp->topo_rports[p]);
      } else if (onp->ln_type == LF_NODE_XBAR) {
	struct lf_xbar *oxp;

	oxp = LF_XBAR(onp);
	off += sprintf(s+off, "x%d, p%d",
	               FMA_XBAR(oxp)->node_id, xp->topo_rports[p]);

	if (FMA_XBAR(oxp)->mf_xbar_resolved) {
	  off += sprintf(s+off, " (resolved)");
	}
      } else {
        off += sprintf(s+off, "weird node type = %d", onp->ln_type);
      }

      fma_log(s);
    }
    fma_log("");
  }
}

static void
dump_map()
{
  struct fma_map_progress *mpp;
  struct fma_nic *fnicp;

  mpp = A.myri->map_progress;

  dump_xbar_list(&mpp->resolved_xbar_anchor, "resolved");
  dump_xbar_list(&mpp->resolving_xbar_anchor, "resolving");
  dump_xbar_list(&mpp->frontier_xbar_anchor, "frontier");
  dump_xbar_list(&mpp->nascent_xbar_anchor, "nascent");
  
  /* print all the NIC-NIC links */
  for (fnicp=mpp->nic_anchor.mf_next;
       fnicp != &mpp->nic_anchor;
       fnicp=fnicp->mf_next) {
    struct lf_nic *nicp;
    struct lf_nic *onicp;
    int p;

    nicp = fnicp->parent;

    for (p=0; p<nicp->num_ports; ++p) {
      onicp = LF_NIC(nicp->topo_ports[p]);
      if (onicp != NULL && onicp->ln_type == LF_NODE_NIC) {
	fma_log(LF_MAC_FORMAT ":p%d - " LF_MAC_FORMAT ":p%d",
	    LF_MAC_ARGS(nicp->mac_addr), p,
	    LF_MAC_ARGS(onicp->mac_addr), nicp->topo_rports[p]);
      }
    }
  }
}

/*
 * Mapping is complete, turn it into a topo map, call completion routine,
 * and then free everything.
 */
static void
fma_mf_mapping_complete()
{
  struct fma_map_progress *mpp;

  mpp = A.myri->map_progress;

  if (mf_debug) {
    dump_map();
  }

  fma_log("Mapping took %d seconds", (int)(time(NULL) - mpp->mf_start_time));

  if (A.debug) {
    if (mpp->mf_outstanding_scouts != 0) {
      fma_log("\toutstanding=%d", mpp->mf_outstanding_scouts);
abort();
    }
  }

  fma_mf_make_topo_map();		/* make a topo map */

  /* call completion routine.  this must free or keep the topo map */
  mpp->map_complete_rtn(mpp->topo_map, mpp->topo_map_size);
  mpp->topo_map = NULL;

  fma_mf_free_mapping();
}

/*
 * create a topo map from the map we just generated
 */
static void
fma_mf_make_topo_map()
{
  struct lf_topo_map *topo_map;
  struct fma_map_progress *mpp;
  struct lf_fabric *fp;
  int topo_map_size;
  int rc;

  mpp = A.myri->map_progress;

  /* turn our mapping structs into a fabric */
  fp = fma_mf_create_fabric();
  if (fp == NULL) {
    LF_ERROR(("Error creating fabric"));
  }

  /* Generate Myrinet Aggregation IDs for nodes that need them */
  lf_lag_assign_mag_ids(fp);

  /* perform some fabric processing */
  rc = lf_process_fabric_for_topo(fp);
  if (rc == -1) LF_ERROR(("Error processing fabric"));

  /* assign map distributors */
  fma_assign_dist_targets(fp);

  /* turn this fabric into a topo map, then immediately free the fabric */
  topo_map = lf_create_topo_map(fp, &topo_map_size);
  lf_free_fabric(fp);
  if (topo_map == NULL) {
    LF_ERROR(("Error generating topo map"));
  }

  /* save topo map info */
  mpp->topo_map = topo_map;
  mpp->topo_map_size = topo_map_size;

  if (A.debug) {
    fma_log("new map has %d nics, %d xbars, %d links",
	    ntohl(topo_map->num_nics_32),
	    ntohl(topo_map->num_xbars_32),
	    ntohl(topo_map->num_links_32));
  }

  /* fill in map header with local info */
  topo_map->map_id_32 = htonl(A.myri->map_session_id);
  LF_MAC_COPY(topo_map->mapper_mac_addr, A.myri->my_max_mac);
  topo_map->mapper_level_16 = htons(A.stand->my_level);

  return;

 except:
  fma_perror_exit(1);
}

/*
 * Send the map we just created to the FMS
 */
void
fma_mf_send_map_to_fms(
  struct lf_topo_map *topo_map,
  int topo_map_size)
{
  struct fms_msg_header hdr;
  int rc;

  /* mapping is no now longer active */
  A.stand->mapping_in_progress = FALSE;

  /* build header */
  hdr.msg_type_32 = htonl(FMA_FMS_MAP_UPLOAD);
  hdr.length_32 = htonl(topo_map_size);

  /* send the header and the message */
  rc = fma_fms_send(FMA_FMS_MAP_UPLOAD, topo_map, topo_map_size, TRUE,
      	fma_free, topo_map);
  if (rc != 0) LF_ERROR(("error sending topo map"));


  return;

 except:
  LF_FREE(topo_map);	/* all done with map */
  fma_perror();		/* basically print error and ignore it */
}

/*
 * Halt a map in progress and free all memory associated with it
 */
void
fma_mf_cancel_mapping()
{
  fma_log("Cancelling mapping");

  /* Only work to do if a mapping actually in progress */
  if (A.myri->map_progress != NULL) {
    fma_mf_free_mapping();	/* free all the data for the mapping process */
  }

  A.stand->mapping_in_progress = FALSE;
}

/*
* Release all memory associated with mapping the fabric
*/
static void
fma_mf_free_mapping()
{
  struct fma_map_progress *mpp;
  struct fma_xbar *anchor;
  struct fma_xbar *fxp;
  struct fma_nic *fnicp;
  struct fma_nic *nfnicp;
  int h;

  if (mf_debug) fma_log("Freeing mapping");

  mpp = A.myri->map_progress;

  /* Cancel scout timer if active */
  if (mpp->mf_scout_timer != NULL) {
    fma_sync_timer_cancel(mpp->mf_scout_timer);
    mpp->mf_scout_timer = NULL;
  }

  /* free all resolved xbars */
  anchor = &mpp->resolved_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=anchor->mf_next) {
    fma_mf_destroy_xbar(fxp->parent);
  }

  anchor = &mpp->resolving_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=anchor->mf_next) {
    fma_mf_destroy_xbar(fxp->parent);
  }

  anchor = &mpp->frontier_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=anchor->mf_next) {
    fma_mf_destroy_xbar(fxp->parent);
  }

  anchor = &mpp->nascent_xbar_anchor;
  for (fxp=anchor->mf_next; fxp != anchor; fxp=anchor->mf_next) {
    fma_mf_destroy_xbar(fxp->parent);
  }

  /* free all the NICs */
  for (fnicp = mpp->nic_anchor.mf_next;
       fnicp != &mpp->nic_anchor;
       fnicp = nfnicp) {
    struct lf_nic *nicp;

    nfnicp = fnicp->mf_next;
    nicp = fnicp->parent;

    /* Only free dangling NICs with no owning host */
    if (nicp->host == NULL) {
      lf_free_nic(nicp);
    }
  }

  /* free any hosts - this will also free their NICs */
  if (mpp->mf_hosts != NULL) {
    for (h=0; h<mpp->num_hosts; ++h) {
      lf_free_host(mpp->mf_hosts[h]);
    }
    LF_FREE(mpp->mf_hosts);
  }
  LF_FREE(mpp->my_nics);
  LF_FREE(mpp->xbar_forward_addresses);
  LF_FREE(mpp->xbar_forward_deltas);
  LF_FREE(A.myri->map_progress);
  return;
}

/*
 * Take the linked lists we have and turn them into a nice, friendly fabric
 */
static struct lf_fabric *
fma_mf_create_fabric()
{
  struct fma_map_progress *mpp;
  struct lf_fabric *fp;
  struct fma_xbar *fxp;
  struct lf_xbar *xp;
  struct lf_host *hp;
  int num_xbars;
  int h;
  int x;

  mpp = A.myri->map_progress;

  LF_CALLOC(fp, struct lf_fabric, 1);

  /*
   * Steal the hosts array
   */
  fp->num_hosts = mpp->num_hosts;
  fp->hosts = mpp->mf_hosts;
  mpp->mf_hosts = NULL;

  /* Copy fma_flags and fw_type from NICs */
  for (h=0; h<fp->num_hosts; ++h) {
    hp = fp->hosts[h];
    hp->fma_flags = FMA_NIC(hp->nics[0])->fma_flags;
    hp->fw_type = FMA_NIC(hp->nics[0])->fw_type;
  }

  /*
   * we need to make 2 passes through the fabric - once to count xbars
   * so we can allocate space, then once more to copy things into the
   * fabric struct
   */

  /* count and mark all xbars - easier than calling realloc */
  num_xbars = 0;
  for (fxp=mpp->resolved_xbar_anchor.mf_next;
       fxp != &mpp->resolved_xbar_anchor;
       fxp=fxp->mf_next) {
    ++num_xbars;
  }

  /* fill in all the xbars */
  fp->num_xbars = num_xbars;
  LF_CALLOC(fp->xbars, struct lf_xbar *, num_xbars);

  /*
   * loop through all xbars, moving them into fabric
   * As a side-effect of this, we're also going to empty the resolved
   * xbar list.
   */
  x = 0;

  for (fxp=mpp->resolved_xbar_anchor.mf_next;
       fxp != &mpp->resolved_xbar_anchor;
       fxp=fxp->mf_next) {

    xp = fxp->parent;
    fp->xbars[x] = xp;
    ++x;
  }

  /* empty the list so we don't end up with double frees */
  mpp->resolved_xbar_anchor.mf_next = &mpp->resolved_xbar_anchor;
  mpp->resolved_xbar_anchor.mf_prev = &mpp->resolved_xbar_anchor;

  return fp;

 except:
  fma_perror_exit(1);
  return NULL;
}

/*
 * Fill in Myrinet packet for a NIC mapping scout
 */
static void
fma_mf_fill_xbar_map_scout(
  struct fma_xbar_scout_pkt *pkt,
  int node_index,
  int is_tagged)
{
  pkt->hdr.type_16 = htons(FMA_PACKET_TYPE);
  if (is_tagged) {
    pkt->hdr.subtype_16 = htons(FMA_SUBTYPE_TAGGED_XBAR_MAP);
  } else {
    pkt->hdr.subtype_16 = htons(FMA_SUBTYPE_XBAR_MAP);
  }

  /* Fill in our info */
  pkt->node_index_32 = htonl(node_index);
  pkt->serial_32 = htonl(A.myri->map_session_id);

  memset(&pkt->tagged_xbar_insert, 0, sizeof(struct lf_tagged_xbar_insert));
}

/*
 * Fill in Myrinet packet for a NIC mapping scout
 */
static void
fma_mf_fill_nic_map_scout(
  struct myri_nic_scout *pkt,
  int node_index,
  unsigned char *route,
  int route_len)
{
  struct fma_nic_map_scout_opaque_data *sodp;

  sodp = (struct fma_nic_map_scout_opaque_data *) &pkt->opaque_scout_data;

  pkt->hdr.type_16 = htons(MYRI_PACKET_TYPE);
  pkt->hdr.subtype_16 = htons(MYRI_SUBTYPE_NIC_SCOUT);

  /* fill in opaque fields */
  sodp->scout_type_8 = A.map_info->first_mapping ? FMA_NST_MAP : FMA_NST_REMAP;
  LF_MAC_COPY(sodp->origin_max_mac_addr, A.myri->my_max_mac);
  sodp->fma_protocol_16 = htons(FMA_PROTOCOL_VERSION);

  /* Fill in our mapping info */
  sodp->level_8 = A.stand->my_level;
  sodp->node_index_32 = htonl(node_index);
  sodp->serial_32 = htonl(A.myri->map_session_id);

  /* Fill in the generic parts */
  fma_fill_nic_generic_scout(pkt, route, route_len);
}

void
fma_mf_free_nic(
  struct lf_nic *nicp)
{
  struct fma_nic *fnicp;

  fnicp = FMA_NIC(nicp);

  /* If this is on some list, remove it */
  if (fnicp->mf_next != NULL) {
    fnicp->mf_prev->mf_next = fnicp->mf_next;
    fnicp->mf_next->mf_prev = fnicp->mf_prev;
  }

  LF_FREE(fnicp->mf_info);
}
